import json
import os
import signal
import tempfile

from grut.libs.proto.objects import conversion_source_pb2
from grut.libs.proto.objects.autogen import schema_pb2
from yt.wrapper import ypath

from crypta.lib.python import (
    grut,
    time_utils,
)
from crypta.lib.python.lb_pusher import logbroker
from crypta.lib.python.lb_pusher.reader_config import ReaderConfig
from crypta.lib.python.solomon import reporter
from crypta.lib.python.yt import yt_helpers
from crypta.s2s.lib import serializers
from crypta.s2s.lib.conversion_source_client import ConversionSourceClient
from crypta.s2s.lib.proto.process_command_pb2 import TProcessCommand
from crypta.s2s.services.conversions_downloader.lib import (
    ftp_downloader,
    google_sheets_downloader,
    link_downloader,
    sftp_downloader,
)
from crypta.s2s.services.conversions_downloader.lib.encrypter import Encrypter


EProcessingStatus = conversion_source_pb2.TConversionSourceSpec.TProcessingInfo.EProcessingStatus
ETypeCode = conversion_source_pb2.TConversionSourceMetaBase.ETypeCode


def run(config, logger, as_service=True):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    yt_client.create("map_node", config.DownloadDir, recursive=True, ignore_existing=True)

    conversion_source_client = ConversionSourceClient(grut.get_client_from_proto(config.Grut), logger)

    pq_client = logbroker.PQClient(
        config.Logbroker.Url,
        config.Logbroker.Port,
        tvm_id=config.Tvm.SourceTvmId,
        tvm_secret=config.Tvm.Secret,
    )
    pq_reader = pq_client.get_reader(ReaderConfig(topic=config.DownloadLogTopic, client_id=config.Consumer, read_only_local=False))
    pq_writer = pq_client.get_writer(config.ProcessLogTopic)
    solomon_reporter = reporter.create_solomon_reporter(
        project=config.Solomon.Project,
        cluster=config.Solomon.Cluster,
        service=config.Solomon.Service,
        url=config.Solomon.Url,
        oauth_token=config.Solomon.Token,
    )

    google_api_key = json.loads(config.GoogleApiKey)
    encrypter = Encrypter(config.DirectEncryptionSecret.encode())

    processor = Processor(conversion_source_client, yt_client, config.DownloadDir, pq_writer, google_api_key, encrypter, solomon_reporter, logger)

    service = Service()
    signal.signal(signal.SIGTERM, service.stop)

    with pq_client, pq_reader, pq_writer, conversion_source_client.object_api_client:
        while service.running:
            for serialized_command in pq_reader.get_messages():
                try:
                    command = serializers.deserialize_download_command(serialized_command)
                    processor.process(command)
                except Exception:
                    logger.exception("")

            pq_reader.commit()

            if not as_service:
                service.stop()


class Processor:
    def __init__(self, conversion_source_client, yt_client, output_dir, pq_writer, google_api_key, encrypter, solomon, logger):
        self.conversion_source_client = conversion_source_client
        self.yt_client = yt_client
        self.output_dir = output_dir
        self.pq_writer = pq_writer
        self.google_api_key = google_api_key
        self.encrypter = encrypter
        self.solomon = solomon
        self.logger = logger

    def process(self, download_command):
        with tempfile.TemporaryDirectory() as tmp_dir:
            meta = download_command.ConversionSource.meta

            self.logger.info("[%s] Start processing...", meta.id)

            filename = "{}_{}".format(meta.id, download_command.Timestamp)
            local_path = os.path.join(tmp_dir, filename)
            yt_path = ypath.ypath_join(self.output_dir, filename)

            try:
                downloader = self.get_downloader(download_command.ConversionSource)
                if downloader is None:
                    self.logger.warn("[%s] Unsupported downloader for %s", meta.id, meta.type_code)
                    return

                self.grut_on_processing_start(meta)

                self.logger.info("[%s] Download file to %s", meta.id, local_path)
                downloader.download(local_path)

                self.logger.info("[%s] Upload file to %s", meta.id, yt_path)
                with open(local_path, "rb") as f:
                    self.yt_client.write_file(yt_path, f)

                self.logger.info("[%s] Issue file processing", meta.id)
                process_command = make_process_command(download_command.ConversionSource, filename)
                self.pq_writer.write(serializers.serialize_process_command(process_command))
                self.solomon.set_value("download_success", 1, labels={"client_id": str(meta.id)})

            except Exception:
                self.logger.exception("")
                self.grut_on_processing_error(meta)
                self.solomon.set_value("download_errors", 1, labels={"client_id": str(meta.id)})

            self.logger.info("[%s] Finish processing...", meta.id)

    def get_downloader(self, conversion_source):
        spec = conversion_source.spec
        if (type_code := conversion_source.meta.type_code) == ETypeCode.TC_FTP:
            return ftp_downloader.FtpDownloader(spec.ftp, self.encrypter, self.logger)
        elif type_code == ETypeCode.TC_LINK:
            return link_downloader.LinkDownloader(spec.link, self.logger)
        elif type_code == ETypeCode.TC_GOOGLE_SHEETS:
            return google_sheets_downloader.GoogleSheetsDownloader(spec.google_sheets, self.google_api_key, self.logger)
        elif type_code == ETypeCode.TC_SFTP:
            return sftp_downloader.SftpDownloader(spec.sftp, self.encrypter, self.logger)
        else:
            return None

    def grut_on_processing_start(self, meta):
        self.conversion_source_client.update_bulk_uniformly(
            [schema_pb2.TConversionSource(
                meta=meta,
                spec=schema_pb2.TConversionSourceSpec(
                    processing_info=schema_pb2.TConversionSourceSpec.TProcessingInfo(
                        processing_status=EProcessingStatus.PS_PROCESSING,
                        last_start_time=time_utils.get_current_time(),
                    ),
                ),
            )],
            set_paths=[
                "/spec/processing_info/processing_status",
                "/spec/processing_info/last_start_time",
            ],
        )

    def grut_on_processing_error(self, meta):
        self.conversion_source_client.update_bulk_uniformly(
            [schema_pb2.TConversionSource(
                meta=meta,
                spec=schema_pb2.TConversionSourceSpec(
                    processing_info=schema_pb2.TConversionSourceSpec.TProcessingInfo(
                        processing_status=EProcessingStatus.PS_ERROR,
                    ),
                ),
            )],
            set_paths=[
                "/spec/processing_info/processing_status",
            ],
        )


class Service:
    def __init__(self):
        self.running = True

    def stop(self, *args):
        self.running = False


def make_process_command(conversion_source, filename):
    conversion_source_copy = schema_pb2.TConversionSource()
    conversion_source_copy.CopyFrom(conversion_source)

    for field in ("ftp", "link", "google_sheets", "sftp"):
        conversion_source_copy.spec.ClearField(field)

    return TProcessCommand(ConversionSource=conversion_source_copy, Filename=filename)
