import csv
import os

from yt import yson

from crypta.s2s.services.conversions_processor.lib.parsers.crm_api_row_parser import CrmApiRowParser
from crypta.s2s.services.conversions_processor.lib.parsers.fields import Fields
from crypta.s2s.services.conversions_processor.lib.parsers import parsing_error


class CrmApiCsvParser:
    delimiter = ";"
    fieldnames = (
        Fields.create_date_time,
        Fields.id_,
        Fields.client_uniq_id,
        Fields.client_ids,
        Fields.emails,
        Fields.phones,
        Fields.emails_md5,
        Fields.phones_md5,
        Fields.order_status,
        Fields.revenue,
        Fields.cost,
    )
    known_fieldnames = set(fieldnames)
    id_fieldnames = {
        Fields.client_ids,
        Fields.emails,
        Fields.phones,
        Fields.emails_md5,
        Fields.phones_md5,
    }

    def __init__(self, order_status_to_revenue, min_create_date_time):
        self.order_status_to_revenue = order_status_to_revenue
        self.crm_api_row_parser = CrmApiRowParser(min_create_date_time)

    def parse(self, source_path, destination_path, errors_path):
        """
        Returns `True` if there is at least 1 valid row with data in `source_file`.
        Writes valid rows to `destination_path` and invalid rows to `errors_path`.
        Normalizes identifiers and modifies revenue.
        """
        has_valid_rows = False

        with open(source_path) as source_fileobj, open(destination_path, "w") as destination_fileobj, open(errors_path, "wb") as errors_fileobj:

            def write_error(obj, what, line):
                yson.dump([{"object": obj, "what": what, "line": line}], errors_fileobj, yson_format="text", yson_type="list_fragment")

            try:
                reader = csv.DictReader(source_fileobj, delimiter=self.delimiter)
                self._check_fieldnames(reader.fieldnames)

                writer = csv.DictWriter(destination_fileobj, fieldnames=self.fieldnames, delimiter=self.delimiter)
                writer.writeheader()
                for index, row in enumerate(reader):
                    try:
                        modified_row = self.crm_api_row_parser.parse(row)
                        self._modify_revenue(modified_row)
                        writer.writerow(modified_row)
                        has_valid_rows = True
                    except parsing_error.ParsingError as e:
                        row_with_known_fields = {k: v for k, v in row.items() if k in self.fieldnames}
                        write_error(row_with_known_fields, str(e), index + 2)

            except csv.Error as e:
                has_valid_rows = False
                write_error(os.path.basename(source_path), str(e), None)

            except UnicodeDecodeError as e:
                has_valid_rows = False
                write_error(os.path.basename(source_path), str(e), None)

            except parsing_error.InvalidFieldnames as e:
                write_error(os.path.basename(source_path), str(e), 1)

        return has_valid_rows

    def _modify_revenue(self, row):
        order_status = row[Fields.order_status]
        revenue = row.get(Fields.revenue)
        row[Fields.revenue] = self.order_status_to_revenue.get(order_status, revenue)

    def _check_fieldnames(self, fieldnames):
        if Fields.create_date_time not in fieldnames:
            raise parsing_error.InvalidFieldnames(f"Missing '{Fields.create_date_time}' field name")

        if not any(fieldname in self.id_fieldnames for fieldname in fieldnames):
            raise parsing_error.InvalidFieldnames("Missing field names for identifiers")

        for fieldname in fieldnames:
            if fieldname not in self.known_fieldnames:
                raise parsing_error.InvalidFieldnames(f"Field names contain unknown field '{fieldname}'")
