from crypta.s2s.services.conversions_processor.lib.parsers.cid_parser import CidParser
from crypta.s2s.services.conversions_processor.lib.parsers.date_time_parser import DateTimeParser
from crypta.s2s.services.conversions_processor.lib.parsers.fields import Fields
from crypta.s2s.services.conversions_processor.lib.parsers.order_statuses import OrderStatuses
from crypta.s2s.services.conversions_processor.lib.parsers.parsing_error import ParsingError


class CrmApiRowParser:
    inside_column_delimiter = ","
    cid_field_to_parser = {
        Fields.client_ids: CidParser.parse_client_id,
        Fields.emails: CidParser.parse_email,
        Fields.phones: CidParser.parse_phone,
        Fields.emails_md5: CidParser.parse_email_md5,
        Fields.phones_md5: CidParser.parse_phone_md5,
    }

    def __init__(self, min_create_date_time):
        self.min_create_date_time = min_create_date_time

    def parse(self, row):
        result_row = {}

        create_date_time = DateTimeParser.parse(self._parse_required_column(row, Fields.create_date_time))

        if create_date_time < self.min_create_date_time:
            raise ParsingError("Row is too old")

        result_row[Fields.create_date_time] = create_date_time.strftime("%Y-%m-%d %H:%M:%S")

        for cid_field, parser in self.cid_field_to_parser.items():
            result_row[cid_field] = self.inside_column_delimiter.join(self._parse_cids(parser, row.get(cid_field)))

        if not any(result_row.get(cid_field) for cid_field in self.cid_field_to_parser.keys()):
            raise ParsingError("Missing field with identifiers")

        for field in (Fields.id_, Fields.client_uniq_id):
            result_row[field] = row.get(field)

        for field in (Fields.revenue, Fields.cost):
            result_row[field] = self._parse_optional_float(row, field)

        result_row[Fields.order_status] = self._parse_order_status(row.get(Fields.order_status))

        return result_row

    @classmethod
    def _parse_cids(cls, parser, s):
        if not s:
            return []

        return [parser(id_value) for id_value in s.split(cls.inside_column_delimiter)]

    @staticmethod
    def _parse_required_column(row, column):
        value = row.get(column)

        if value is None:
            raise ParsingError(f"Missing column '{column}'")

        if not value:
            raise ParsingError(f"Empty value of column '{column}'")

        return value

    @staticmethod
    def _parse_optional_float(row, column):
        value = row.get(column)

        if not value:
            return None

        try:
            return str(float(value))
        except ValueError:
            raise ParsingError(f"Invalid '{column}' value '{value}'")

    @staticmethod
    def _parse_order_status(s):
        if s and (s not in {OrderStatuses.paid, OrderStatuses.in_progress, OrderStatuses.cancelled, OrderStatuses.spam}):
            raise ParsingError(f"Invalid 'order_status' value '{s}'")

        return s or OrderStatuses.paid
