import itertools
import logging
import os
import urllib.parse

import requests
from requests.status_codes import codes
import retry


logger = logging.getLogger(__name__)


def prepare_csv_parts(filepath, lines_per_file):
    with open(filepath, "r") as f:
        header = next(f)
        for key, lines in itertools.groupby(enumerate(f), key=lambda enumerated_line: enumerated_line[0] // lines_per_file):
            yield f"{os.path.basename(filepath)}-{key}", f"{header}{''.join(line for i, line in lines)}"


class CdpApiClient:
    tvm_alias = "CdpApi"

    def __init__(self, address, tvm_client, lines_per_file, logger):
        self.session = requests.Session()
        self.tvm_client = tvm_client
        self.logger = logger
        self.simple_orders_url_template = urllib.parse.urljoin(address, "/cdp/internal/v1/upload_for_user/counter/{}/data/simple_orders")
        self.lines_per_file = lines_per_file

    def simple_orders(self, filepath, counter_id, uid, metrics):
        url = self.simple_orders_url_template.format(counter_id)
        headers = {
            "X-Ya-Service-Ticket": self.tvm_client.get_service_ticket_for(self.tvm_alias),
        }
        params = {"uid": uid, "merge_mode": "APPEND", "delimiter_type": "semicolon"}
        result = True

        for filename, body in prepare_csv_parts(filepath, self.lines_per_file):
            try:
                result = result and self.request_simple_orders(url=url, filename=filename, body=body, params=params, headers=headers, metrics=metrics)
            except Exception:
                logger.exception("Can't upload orders. Exhausted all restries")
                result = False

        if metrics is not None:
            metrics.log_upload(result)

        return result

    @retry.retry((requests.ConnectionError, requests.HTTPError, requests.Timeout), tries=3, delay=1, backoff=2, logger=logger)
    def request_simple_orders(self, url, filename, body, params, headers, metrics):
        self.logger.info("Request '%s' with params '%s' and filename '%s'", url, params, filename)
        if metrics is not None:
            metrics.log_simple_orders_call()

        files = {"file": (filename, body, "text/csv")}
        response = self.session.post(url, files=files, params=params, headers=headers)

        self.logger.info("CRM API returns '%s'", response.status_code)

        if metrics is not None:
            metrics.log_simple_orders_elapsed_time(response.elapsed.total_seconds())
            metrics.log_simple_orders_http_status_code(response.status_code)

        if response.status_code in {codes.bad_request, codes.forbidden}:
            return False
        elif not response.ok:
            response.raise_for_status()

        return True
