import datetime
import os
import tempfile

from frozendict import frozendict
from grut.libs.proto.objects import conversion_source_pb2
from grut.libs.proto.objects.autogen import schema_pb2
import yt.orm.library.common as orm

from crypta.lib.python import time_utils
from crypta.s2s.lib import meta_injecting_filter
from crypta.s2s.services.conversions_processor.lib.processor import stats
from crypta.s2s.services.conversions_processor.lib.parsers.crm_api_csv_parser import CrmApiCsvParser

EProcessingStatus = conversion_source_pb2.TConversionSourceSpec.TProcessingInfo.EProcessingStatus


class Processor:
    def __init__(self, conversion_source_client, yt_manipulator, cdp_api_client, max_order_age, processing_metrics, logger):
        self.conversion_source_client = conversion_source_client
        self.yt_manipulator = yt_manipulator
        self.cdp_api_client = cdp_api_client
        self.max_order_age = max_order_age
        self.processing_metrics = processing_metrics
        self.logger = logger

    def process(self, process_command, labels):
        spec = process_command.ConversionSource.spec
        destination = spec.WhichOneof("destination")
        labels[stats.LabelNames.destination] = destination

        if destination != "crm_api_destination":
            self.logger.warn("Unsupported destination")
            return False

        metrics = self.processing_metrics[frozendict(labels)]

        meta = process_command.ConversionSource.meta
        filename = process_command.Filename

        with meta_injecting_filter.ctx(meta):
            with tempfile.TemporaryDirectory() as tmp_dir, self.yt_manipulator.yt_client.Transaction():
                self.logger.info("Start processing...")

                local_path = os.path.join(tmp_dir, filename)
                self.yt_manipulator.download(filename, local_path)
                self.yt_manipulator.backup_downloaded(meta, filename)
                self.yt_manipulator.remove_downloaded(filename)

                if self.upload_csv(meta, local_path, spec.conversion_actions, spec.crm_api_destination, metrics):
                    self.grut_on_processing_success(meta)
                else:
                    self.grut_on_processing_error(meta)

            self.logger.info("Finish processing...")

    def grut_on_processing_success(self, meta):
        self._update_processing_status(meta, EProcessingStatus.PS_SUCCESS)

    def grut_on_processing_error(self, meta):
        self._update_processing_status(meta, EProcessingStatus.PS_ERROR)

    def upload_csv(self, meta, path, conversion_actions, crm_api_destination, metrics):
        order_status_to_revenue = {
            conversion_action.name: conversion_action.fixed.cost
            for conversion_action in conversion_actions
            if conversion_action.WhichOneof("value") == "fixed"
        }
        to_upload_path = f"{path}-upload"
        errors_path = f"{path}-errors"

        min_create_date_time = datetime.datetime.utcfromtimestamp(time_utils.get_current_time()) - self.max_order_age
        has_valid_rows = CrmApiCsvParser(order_status_to_revenue, min_create_date_time).parse(path, to_upload_path, errors_path)
        self.yt_manipulator.upload_errors(meta, errors_path)

        metrics.log_parsing(has_valid_rows)

        if not has_valid_rows:
            return False

        self.yt_manipulator.backup_uploaded(meta, to_upload_path)
        return self.cdp_api_client.simple_orders(
            filepath=to_upload_path,
            counter_id=crm_api_destination.counter_id,
            uid=crm_api_destination.access_uid,
            metrics=metrics,
        )

    def _update_processing_status(self, meta, processing_status):
        try:
            self.conversion_source_client.update_bulk_uniformly(
                [schema_pb2.TConversionSource(
                    meta=meta,
                    spec=schema_pb2.TConversionSourceSpec(
                        processing_info=schema_pb2.TConversionSourceSpec.TProcessingInfo(
                            processing_status=processing_status,
                        ),
                    ),
                )],
                set_paths=[
                    "/spec/processing_info/processing_status",
                ],
            )
        except orm.NoSuchObjectError:
            self.logger.exception("")
