import http.client

from library.python.monlib import metric_registry

from crypta.lib.python.worker_utils import task_metrics


class Sensors:
    parsing_success = "parsing_success"
    parsing_failure = "parsing_failure"

    upload_success = "upload_success"
    upload_failure = "upload_failure"

    simple_orders_call = "simple_orders_call"
    simple_orders_elapsed_time = "simple_orders_elapsed_time"


class LabelNames:
    destination = "destination"
    category = "category"
    sensor = "sensor"


class Categories:
    http_status_code = "http_status_code"


def get_metrics():
    return [{
        LabelNames.destination: "crm_api_destination",
    }, {
        LabelNames.destination: "post_back_destination",
    }]


class ProcessingMetrics:
    status_code_unknown = "unknown"

    def __init__(self, mp_metric_registry, labels):
        self.parsing_success = mp_metric_registry.rate(task_metrics.make_sensor(Sensors.parsing_success, labels))
        self.parsing_failure = mp_metric_registry.rate(task_metrics.make_sensor(Sensors.parsing_failure, labels))
        self.upload_success = mp_metric_registry.rate(task_metrics.make_sensor(Sensors.upload_success, labels))
        self.upload_failure = mp_metric_registry.rate(task_metrics.make_sensor(Sensors.upload_failure, labels))
        self.simple_orders_call = mp_metric_registry.rate(task_metrics.make_sensor(Sensors.simple_orders_call, labels))
        self.simple_orders_elapsed_time = mp_metric_registry.histogram_rate(
            task_metrics.make_sensor(Sensors.simple_orders_elapsed_time, labels),
            metric_registry.HistogramType.Explicit,
            buckets=(1, 2, 5, 10, 20, 50, 100),
        )

        http_status_code_labels = {LabelNames.category: Categories.http_status_code, **labels}
        self.simple_orders_http_status_code = {
            status_code: mp_metric_registry.rate(task_metrics.make_sensor(status_code, http_status_code_labels))
            for status_code in [self.status_code_unknown] + [str(item.value) for item in http.client.responses.keys()]
        }

    def log_parsing(self, success):
        (self.parsing_success if success else self.parsing_failure).inc()

    def log_upload(self, success):
        (self.upload_success if success else self.upload_failure).inc()

    def log_simple_orders_call(self):
        self.simple_orders_call.inc()

    def log_simple_orders_elapsed_time(self, time):
        self.simple_orders_elapsed_time.collect(time)

    def log_simple_orders_http_status_code(self, status_code):
        status_code = str(status_code)
        if status_code not in self.simple_orders_http_status_code:
            status_code = self.status_code_unknown

        self.simple_orders_http_status_code[status_code].inc()
