import os

from yt.wrapper import ypath

from crypta.lib.python.yt import yt_helpers


class YtManipulator:
    def __init__(self, yt_client, download_dir, downloaded_backup_dir, uploaded_backup_dir, errors_dir, max_backup_size, logger):
        self.yt_client = yt_client
        self.download_dir = download_dir
        self.downloaded_backup_dir = downloaded_backup_dir
        self.uploaded_backup_dir = uploaded_backup_dir
        self.errors_dir = errors_dir
        self.max_backup_size = max_backup_size
        self.logger = logger

    @classmethod
    def from_proto(cls, config, logger):
        yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
        return cls(
            yt_client=yt_client,
            download_dir=config.DownloadDir,
            downloaded_backup_dir=config.DownloadedBackupDir,
            uploaded_backup_dir=config.UploadedBackupDir,
            errors_dir=config.ErrorsDir,
            max_backup_size=config.MaxBackupSize,
            logger=logger,
        )

    def create_directories(self):
        for cypress_path in (self.download_dir, self.downloaded_backup_dir, self.uploaded_backup_dir, self.errors_dir):
            self.yt_client.mkdir(cypress_path, recursive=True)

    def backup_downloaded(self, meta, filename):
        source_path = ypath.ypath_join(self.download_dir, filename)
        destination_dir_path = ypath.ypath_join(self.downloaded_backup_dir, str(meta.id))
        destination_path = ypath.ypath_join(destination_dir_path, filename)

        self.logger.info("Backup downloaded '%s' to '%s'", source_path, destination_path)
        self.yt_client.copy(source_path, destination_path, recursive=True)
        self.keep_last(destination_dir_path)

    def remove_downloaded(self, filename):
        cypress_path = ypath.ypath_join(self.download_dir, filename)
        self.logger.info("Remove downloaded '%s'", cypress_path)
        self.yt_client.remove(cypress_path)

    def download(self, filename, destination_path):
        cypress_path = ypath.ypath_join(self.download_dir, filename)

        self.logger.info("Download downloaded '%s' to '%s'", cypress_path, destination_path)

        with open(destination_path, "wb") as f:
            for chunk in self.yt_client.read_file(cypress_path).chunk_iter():
                f.write(chunk)

    def upload_errors(self, meta, local_path):
        tablename = os.path.basename(local_path)
        dir_cypress_path = ypath.ypath_join(self.errors_dir, str(meta.id))
        cypress_path = ypath.ypath_join(dir_cypress_path, tablename)

        self.logger.info("Upload errors '%s' to '%s'", local_path, cypress_path)
        self.yt_client.mkdir(dir_cypress_path, recursive=True)
        with open(local_path, "rb") as f:
            self.yt_client.write_table(cypress_path, f, format="yson", raw=True)

        self.keep_last(dir_cypress_path)

    def backup_uploaded(self, meta, local_path):
        filename = os.path.basename(local_path)
        dir_cypress_path = ypath.ypath_join(self.uploaded_backup_dir, str(meta.id))
        cypress_path = ypath.ypath_join(dir_cypress_path, filename)
        self.yt_client.mkdir(dir_cypress_path, recursive=True)
        yt_helpers.backup_local_file(local_path, cypress_path, yt_client=self.yt_client)
        self.keep_last(dir_cypress_path)

    def keep_last(self, cypress_path):
        paths = list(reversed(self.yt_client.list(cypress_path, sort=True, absolute=True)))
        for path in paths[self.max_backup_size:]:
            self.logger.info("Remove old path '%s'", path)
            self.yt_client.remove(path)
