import datetime
import functools

from grut.libs.proto.objects.autogen import schema_pb2
from library.python.monlib import metric_registry

from crypta.lib.python import (
    grut,
    time_utils,
)
from crypta.lib.python.lb_pusher import logbroker
from crypta.lib.python.solomon.client import SolomonClient
from crypta.s2s.lib import (
    serializers,
    stats_helpers,
)
from crypta.s2s.lib.conversion_source_client import ConversionSourceClient
from crypta.s2s.lib.proto.download_command_pb2 import TDownloadCommand


S2S_CONVERSION_SOURCE_TYPES = [
    schema_pb2.TConversionSourceMetaBase.ETypeCode.TC_FTP,
    schema_pb2.TConversionSourceMetaBase.ETypeCode.TC_LINK,
    schema_pb2.TConversionSourceMetaBase.ETypeCode.TC_GOOGLE_SHEETS,
    schema_pb2.TConversionSourceMetaBase.ETypeCode.TC_SFTP,
]


def run(config, logger):
    current_datetime = time_utils.get_current_moscow_datetime()
    current_timestamp = int(current_datetime.timestamp())

    solomon = SolomonClient(auth_token=config.Solomon.Token, address=config.Solomon.Url)
    stats_collector = StatsCollector(metric_registry.MetricRegistry(), current_timestamp)

    conversion_source_client = ConversionSourceClient(grut.get_client_from_proto(config.Grut), logger)

    pq_client = logbroker.PQClient(
        config.Logbroker.Url,
        config.Logbroker.Port,
        tvm_id=config.Tvm.SourceTvmId,
        tvm_secret=config.Tvm.Secret,
    )
    pq_writer = pq_client.get_writer(config.DownloadLogTopic)

    try:
        _run(pq_client, pq_writer, conversion_source_client, stats_collector, current_datetime, current_timestamp, config.BatchSize, logger)
    finally:
        utc_datetime = datetime.datetime.utcfromtimestamp(current_timestamp)
        solomon.push_registry(config.Solomon.Project, config.Solomon.Cluster, config.Solomon.Service, stats_collector.registry, timestamp=utc_datetime)


def _run(pq_client, pq_writer, conversion_source_client, stats_collector, current_datetime, current_timestamp, batch_size, logger):
    attribute_selector = [
        "/meta/id",
        "/meta/type_code",
        "/spec/ftp",
        "/spec/link",
        "/spec/google_sheets",
        "/spec/sftp",
        "/spec/conversion_actions",
        "/spec/processing_info/last_scheduled_time",
        "/spec/update_period_hours",
        "/spec/crm_api_destination",
        "/spec/post_back_destination",
    ]
    filter_ = make_filter(current_timestamp)

    with pq_client, pq_writer, conversion_source_client.object_api_client:
        for batch in conversion_source_client.select_all_batched(filter_=filter_, attribute_selector=attribute_selector, limit=batch_size):
            grut_update = [make_conversion_source_update(conversion_source, current_datetime) for conversion_source in batch]
            commands = [make_download_command(conversion_source, current_timestamp) for conversion_source in batch]

            conversion_source_client.update_bulk_uniformly(grut_update, set_paths=["/spec/processing_info/last_scheduled_time"])

            for command in commands:
                logger.info("[%s] Issue downloading...", command.ConversionSource.meta.id)
                pq_writer.write(serializers.serialize_download_command(command))

            for conversion_source in batch:
                stats_collector.collect(conversion_source)


def make_download_command(conversion_source, timestamp):
    conversion_source_copy = schema_pb2.TConversionSource()
    conversion_source_copy.CopyFrom(conversion_source)
    conversion_source_copy.spec.ClearField("processing_info")
    conversion_source_copy.spec.ClearField("update_period_hours")
    return TDownloadCommand(ConversionSource=conversion_source_copy, Timestamp=timestamp)


def generate_last_scheduled_time(conversion_source, current_datetime):
    hour = conversion_source.meta.id % 24

    last_scheduled_datetime = current_datetime.replace(hour=hour, minute=0, second=0, microsecond=0)
    if last_scheduled_datetime > current_datetime:
        last_scheduled_datetime -= datetime.timedelta(days=1)

    return int(last_scheduled_datetime.timestamp())


def make_conversion_source_update(conversion_source, current_datetime):
    conversion_source_copy = schema_pb2.TConversionSource()
    conversion_source_copy.meta.CopyFrom(conversion_source.meta)
    conversion_source_copy.spec.CopyFrom(schema_pb2.TConversionSourceSpec(
        processing_info=schema_pb2.TConversionSourceSpec.TProcessingInfo(
            last_scheduled_time=generate_last_scheduled_time(conversion_source, current_datetime),
        )
    ))
    return conversion_source_copy


def make_filter(current_timestamp):
    type_code_condition = " OR ".join(f"[/meta/type_code] = {type_code}" for type_code in S2S_CONVERSION_SOURCE_TYPES)
    with_update_time_condition = "[/spec/update_period_hours] > 0u OR [/spec/processing_info/last_scheduled_time] = 0u"
    scheduled_time_condition = f"uint64([/spec/processing_info/last_scheduled_time]) + uint64([/spec/update_period_hours]) * 3600 <= {current_timestamp}u"
    return f"({type_code_condition}) AND ({with_update_time_condition}) AND ({scheduled_time_condition})"


class StatsCollector:
    class Labels:
        conversion_source_type = "conversion_source_type"
        sensor = "sensor"

    class Sensors:
        scheduled = "scheduled_count"
        scheduling_lag = "scheduling_lag_seconds"

    def __init__(self, registry, current_timestamp):
        self.registry = registry
        self.scheduling_lag_histogram = self.registry.histogram_counter(
            {self.Labels.sensor: self.Sensors.scheduling_lag},
            metric_registry.HistogramType.Explicit,
            buckets=(600, 1800, 3600, 7200, 14400, 28800, 86400),
        )
        self.current_timestamp = current_timestamp

    @functools.cache
    def _scheduled_counter(self, conversion_source_type_code):
        conversion_source_type = stats_helpers.get_type_code_name(conversion_source_type_code)
        return self.registry.counter({
            self.Labels.conversion_source_type: conversion_source_type,
            self.Labels.sensor: self.Sensors.scheduled,
        })

    def collect(self, conversion_source):
        self._scheduled_counter(conversion_source.meta.type_code).inc()
        next_scheduled_time = conversion_source.spec.processing_info.last_scheduled_time + 3600 * conversion_source.spec.update_period_hours
        self.scheduling_lag_histogram.collect(self.current_timestamp - next_scheduled_time)
