from datetime import timedelta as dt
import json
import os
import tempfile

import gspread
from yt.wrapper import ypath

from crypta.lib.python import time_utils
from crypta.lib.python.yt import yt_helpers
from crypta.s2s.services.transfer_conversions_to_yt.lib import helpers


class GoogleSheetsProviderRunner:
    def __init__(
        self,
        provider,
        yt_client,
        fresh_dir,
        backup_config,
        errors_config,
        parser,
        logger,
    ):
        self.yt_client = yt_client
        self.provider = provider
        self.fresh_dir = fresh_dir
        self.backup_config = backup_config
        self.errors_config = errors_config
        self.parser = parser
        self.logger = logger

    def run(self):
        self.logger.info("Process '%s'", self.provider.Url)

        google_client = gspread.service_account_from_dict(json.loads(self.provider.Key), scopes=gspread.auth.READONLY_SCOPES)
        spreadsheet = google_client.open_by_url(self.provider.Url)
        worksheet = spreadsheet.get_worksheet(0)
        raw_conversions = worksheet.get_all_records()

        now = str(time_utils.get_current_time())
        output_table_name = now
        filename = "{}.json".format(now)

        with tempfile.TemporaryDirectory() as tmpdirname, self.yt_client.Transaction():
            local_path = os.path.join(tmpdirname, filename)

            with open(local_path, "w") as f:
                json.dump(raw_conversions, f)

            backup_cypress_path = ypath.ypath_join(self.backup_config.Dir, filename)
            yt_helpers.backup_local_file(local_path, backup_cypress_path, ttl_timedelta=dt(days=self.backup_config.TtlDays), yt_client=self.yt_client)

            helpers.upload_to_yt(
                yt_client=self.yt_client,
                raw_conversions=raw_conversions,
                parser=self.parser,
                fresh_table=ypath.ypath_join(self.fresh_dir, output_table_name),
                errors_table=ypath.ypath_join(self.errors_config.Dir, output_table_name),
                errors_ttl=dt(days=self.errors_config.TtlDays),
                logger=self.logger,
            )
