import csv

from crypta.lib.python.yt import yt_helpers
from crypta.s2s.lib import schemas
from crypta.s2s.services.transfer_conversions_to_yt.lib import parsers


CSV_EXT = ".csv"


def upload_csv_to_yt(yt_client, local_path, parser, fresh_table, errors_table, errors_ttl, logger):
    logger.info("Parse and upload '%s'", local_path)

    with open(local_path) as f:
        reader = csv.DictReader(f)
        upload_to_yt(yt_client, reader, parser, fresh_table, errors_table, errors_ttl, logger)


def upload_to_yt(yt_client, raw_conversions, parser, fresh_table, errors_table, errors_ttl, logger):
    logger.info("Write valid conversions to '%s'", fresh_table)
    logger.info("Write invalid conversions to '%s'", errors_table)

    valid_conversions = []
    invalid_conversions = []
    for item in raw_conversions:
        parsing_result = parser.parse(item)
        if parsers.is_success(parsing_result):
            valid_conversions.extend(_serialize(proto) for proto in parsing_result.result)
        else:
            invalid_conversions.append({
                "Value": item,
                "ErrorMsg": parsing_result.error_msg,
            })

    yt_client.create("table", fresh_table, recursive=True, attributes={"schema": schemas.get_conversion_schema()})
    yt_client.write_table(fresh_table, valid_conversions, raw=False)

    yt_client.create("table", errors_table, recursive=True, attributes={"schema": schemas.get_raw_conversion_parsing_error_schema()})
    yt_client.write_table(errors_table, invalid_conversions)
    yt_helpers.set_ttl(errors_table, errors_ttl, yt_client=yt_client, remove_if_empty=True)


def _serialize(proto):
    # TODO(kolontaev): Разобраться с сериализацией дабловского нуля в proto2json
    serialized = {
        "Yclid": proto.Yclid,
        "GoalId": proto.GoalId,
        "Timestamp": proto.Timestamp,
    }

    if proto.HasField("Value") and proto.HasField("Currency"):
        serialized["Value"] = proto.Value
        serialized["Currency"] = proto.Currency

    return serialized


def is_csv(filename):
    return filename.endswith(CSV_EXT)
