import datetime
import string

import dateutil.parser
import pytz

from crypta.lib.python import (
    iso4217,
    time_utils,
)
from crypta.s2s.lib.proto.conversion_pb2 import TConversion
from crypta.s2s.services.transfer_conversions_to_yt.lib.default_csv_column_names import DefaultCsvColumnNames


class Success:
    def __init__(self, result):
        self.result = result

    def __repr__(self):
        return "Success(result={})".format(self.result)

    def __eq__(self, other):
        return isinstance(other, Success) and self.result == other.result


def is_success(x):
    return isinstance(x, Success)


class Failure:
    def __init__(self, error_msg):
        self.error_msg = error_msg

    def __repr__(self):
        return 'Failure(error_msg="{}")'.format(self.error_msg)

    def __eq__(self, other):
        return isinstance(other, Failure) and self.error_msg == other.error_msg


def is_failure(x):
    return isinstance(x, Failure)


class ConversionParser:
    def __init__(self, timestamp_range, conversion_name_to_goal_ids=None, static_goal_id=None, column_names=None):
        if (conversion_name_to_goal_ids is None) == (static_goal_id is None):
            raise Exception("You must specify exactly one argument from (conversion_name_to_goal_ids, static_goal_id)")

        self.min_timestamp, self.max_timestamp = timestamp_range
        self.conversion_name_to_goal_ids = conversion_name_to_goal_ids
        self.static_goal_id = static_goal_id
        self.column_names = column_names or {}

    def _column_name(self, default):
        return self.column_names.get(default, default)

    def parse(self, row):
        yclid_column = self._column_name(DefaultCsvColumnNames.yclid)
        conversion_name_column = self._column_name(DefaultCsvColumnNames.conversion_name)
        conversion_time_column = self._column_name(DefaultCsvColumnNames.conversion_time)
        conversion_value_column = self._column_name(DefaultCsvColumnNames.conversion_value)
        conversion_currency_column = self._column_name(DefaultCsvColumnNames.conversion_currency)

        for column in (
            yclid_column,
            conversion_time_column,
        ):
            if is_failure((r := _check_required_column(row, column))):
                return r

        if self.static_goal_id is None:
            if is_failure((r := _check_required_column(row, conversion_name_column))):
                return r

            conversion_name = row[conversion_name_column]
            if conversion_name not in self.conversion_name_to_goal_ids:
                return Failure("Unknown value '{}' of column '{}'".format(conversion_name, conversion_name_column))

            goal_ids = self.conversion_name_to_goal_ids[conversion_name]
        else:
            goal_ids = [self.static_goal_id]

        conversion_time = str(row[conversion_time_column])
        try:
            conversion_timestamp = _parse_conversion_time_to_timestamp(conversion_time)
        except Exception:
            return Failure("Invalid value '{}' of column '{}'".format(conversion_time, conversion_time_column))

        if (conversion_timestamp < self.min_timestamp or conversion_timestamp > self.max_timestamp):
            return Failure("Timestamp '{}' for value '{}' of column '{}' is too old or from future".format(conversion_timestamp, conversion_time, conversion_time_column))

        conversion_value = None
        raw_conversion_value = str(row.get(conversion_value_column, ""))
        if raw_conversion_value:
            try:
                conversion_value = float(raw_conversion_value)
            except Exception:
                return Failure("Invalid value '{}' of column '{}'".format(raw_conversion_value, conversion_value_column))

        conversion_currency = row.get(conversion_currency_column) or None
        if conversion_currency is not None and conversion_currency not in iso4217.CURRENCY_CODES:
            return Failure("Invalid value '{}' of column '{}'".format(conversion_currency, conversion_currency_column))

        def _xor_failure(value, column, missing_or_empty_column):
            return Failure("Non-empty value '{}' of column '{}' while column '{}' is missing or empty".format(value, column, missing_or_empty_column))

        if conversion_value is None and conversion_currency is not None:
            return _xor_failure(conversion_currency, conversion_currency_column, conversion_value_column)

        if conversion_currency is None and conversion_value is not None:
            return _xor_failure(conversion_value, conversion_value_column, conversion_currency_column)

        protos = [TConversion(
            Yclid=str(row[yclid_column]),
            GoalId=goal_id,
            Timestamp=conversion_timestamp,
            Value=conversion_value,
            Currency=conversion_currency,
        ) for goal_id in goal_ids]

        return Success(protos)


def get_current_meaningful_timestamp_range():
    now = time_utils.get_current_time()
    return (now - datetime.timedelta(days=365).total_seconds(), now + datetime.timedelta(days=1).total_seconds())


def _check_required_column(row, column):
    if column not in row:
        return Failure("Missing column '{}'".format(column))
    if not row[column]:
        return Failure("Empty value of column '{}'".format(column))
    return Success(row)


def _parse_conversion_time_to_timestamp(s):
    try:
        return int(s)
    except ValueError:
        dt = _parse_conversion_time_to_datetime(s)

        if dt.tzinfo is None:
            dt = pytz.utc.localize(dt)

        return int(dt.timestamp())


def _parse_conversion_time_to_datetime(s):
    if (s[-1] in string.digits) or (s[-2:] in {"AM", "PM"}):
        return dateutil.parser.parse(s)

    date_string, tz_string = s.rsplit(" ", 1)
    return pytz.timezone(tz_string).localize(dateutil.parser.parse(date_string))
