from datetime import timedelta as dt
import os
import tempfile

import more_itertools
from yt.wrapper import ypath

from crypta.lib.python.sftp.client import SftpClient
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.processed_tracker import ProcessedTracker
from crypta.s2s.services.transfer_conversions_to_yt.lib import helpers


CSV_EXT = ".csv"


class SftpProviderRunner:
    def __init__(
        self,
        provider,
        yt_client,
        fresh_dir,
        backup_config,
        errors_config,
        parser,
        logger,
    ):
        self.yt_client = yt_client
        self.sftp_client = SftpClient.from_proto(provider.SftpConfig)
        self.source_dir = provider.SourceDir
        self.tracker = ProcessedTracker(yt_client, provider.TrackTable)
        self.fresh_dir = fresh_dir
        self.backup_config = backup_config
        self.errors_config = errors_config
        self.parser = parser
        self.logger = logger

    def run(self):
        with self.sftp_client:
            filenames = self.sftp_client.listdir(self.source_dir)
            invalid_filenames, valid_filenames = more_itertools.partition(helpers.is_csv, filenames)
            self.tracker.for_each_unprocessed(valid_filenames, self._process)

        if invalid_filenames := list(invalid_filenames):
            raise Exception("Invalid filenames found: %s", invalid_filenames)

    def _process(self, tx, filename):
        self.logger.info("Process '%s'", filename)
        tmp_dir = tempfile.mkdtemp()
        local_path = os.path.join(tmp_dir, filename)
        remote_path = os.path.join(self.source_dir, filename)
        backup_cypress_path = ypath.ypath_join(self.backup_config.Dir, filename)

        tablename = filename.removesuffix(helpers.CSV_EXT)
        fresh_cypress_path = ypath.ypath_join(self.fresh_dir, tablename)
        errors_cypress_path = ypath.ypath_join(self.errors_config.Dir, tablename)

        try:
            self.sftp_client.download(remote_path, local_path)
            yt_helpers.backup_local_file(local_path, backup_cypress_path, ttl_timedelta=dt(days=self.backup_config.TtlDays), yt_client=self.yt_client)
            helpers.upload_csv_to_yt(
                yt_client=self.yt_client,
                local_path=local_path,
                parser=self.parser,
                fresh_table=fresh_cypress_path,
                errors_table=errors_cypress_path,
                errors_ttl=dt(days=self.errors_config.TtlDays),
                logger=self.logger,
            )

        finally:
            if os.path.exists(local_path):
                os.remove(local_path)

            os.rmdir(tmp_dir)
