import datetime
import os
import tempfile

import requests
from yt.wrapper import ypath

from crypta.lib.python import time_utils
from crypta.lib.python.yt import yt_helpers
from crypta.s2s.services.transfer_conversions_to_yt.lib import helpers


class UrlProviderRunner:
    def __init__(
        self,
        provider,
        yt_client,
        fresh_dir,
        backup_config,
        errors_config,
        parser,
        logger,
    ):
        self.yt_client = yt_client
        self.url = provider.Url
        self.fresh_dir = fresh_dir
        self.backup_config = backup_config
        self.errors_config = errors_config
        self.parser = parser
        self.logger = logger

    def run(self):
        tmp_dir = tempfile.mkdtemp()
        now = str(time_utils.get_current_time())
        local_path = os.path.join(tmp_dir, "{}.csv".format(now))

        try:
            self._download(local_path)

            with self.yt_client.Transaction():
                self._backup(local_path)
                self._upload(local_path, table_name=now)

        finally:
            if local_path is not None and os.path.exists(local_path):
                os.remove(local_path)

            os.rmdir(tmp_dir)

    def _download(self, local_path):
        self.logger.info("Download '%s' to '%s'", self.url, local_path)
        with open(local_path, "wb") as f:
            for chunk in requests.get(self.url).iter_content(4096):
                f.write(chunk)

    def _backup(self, local_path):
        backup_cypress_path = ypath.ypath_join(self.backup_config.Dir, os.path.basename(local_path))
        yt_helpers.backup_local_file(local_path, backup_cypress_path, ttl_timedelta=datetime.timedelta(days=self.backup_config.TtlDays), yt_client=self.yt_client)

    def _upload(self, local_path, table_name):
        helpers.upload_csv_to_yt(
            yt_client=self.yt_client,
            local_path=local_path,
            parser=self.parser,
            fresh_table=ypath.ypath_join(self.fresh_dir, table_name),
            errors_table=ypath.ypath_join(self.errors_config.Dir, table_name),
            errors_ttl=datetime.timedelta(days=self.errors_config.TtlDays),
            logger=self.logger,
        )
