import concurrent.futures as cf
import datetime

import ratelimit
import requests
import retry
from yt.wrapper import ypath

from crypta.lib.python.per_thread_session_pool import PerThreadSessionPool
from crypta.lib.python.yt import yt_helpers
from crypta.s2s.lib import schemas


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    yt_client.create("map_node", config.ToPostbackErrors.Dir, recursive=True, ignore_existing=True)
    yt_client.create("map_node", config.ToPostbackBackup.Dir, recursive=True, ignore_existing=True)

    while True:
        with yt_client.Transaction():
            to_postback_tables = yt_client.list(config.ToPostbackDir, sort=True)

            if not to_postback_tables:
                logger.info("No tables to process")
                break

            name = to_postback_tables[0]
            to_postback_table = ypath.ypath_join(config.ToPostbackDir, name)
            backup_table = ypath.ypath_join(config.ToPostbackBackup.Dir, name)
            errors_table = ypath.ypath_join(config.ToPostbackErrors.Dir, name)

            logger.info("Process '%s'", to_postback_table)

            yt_helpers.make_backup_table(to_postback_table, backup_table, datetime.timedelta(days=config.ToPostbackBackup.TtlDays), yt_client=yt_client)

            yt_client.create("table", errors_table, recursive=True, ignore_existing=True, attributes={"schema": schemas.get_conversion_upload_error_schema()})

            logger.info("Sending table '%s' to postback", to_postback_table)
            logger.info("Save errors to '%s'", errors_table)

            rows = yt_client.read_table(to_postback_table)
            errors = _upload(rows, config.Postback, logger)
            yt_client.write_table(errors_table, errors)
            yt_helpers.set_ttl(errors_table, datetime.timedelta(days=config.ToPostbackErrors.TtlDays), yt_client=yt_client, remove_if_empty=True)

            logger.info("Remove '%s'", to_postback_table)
            yt_client.remove(to_postback_table)


def _upload(rows, postback_config, logger):
    session_pool = PerThreadSessionPool()

    retry_config = postback_config.Retry

    @retry.retry(
        tries=retry_config.Tries,
        delay=retry_config.Delay,
        max_delay=retry_config.MaxDelay,
        backoff=retry_config.Backoff,
        jitter=retry_config.Jitter,
        logger=logger,
    )
    @ratelimit.sleep_and_retry
    @ratelimit.limits(
        calls=postback_config.MaxRps,
        period=1,
    )
    def request_unsafe(params):
        session = session_pool.get_session()
        return session.get(postback_config.Url, params=params).raise_for_status()

    def request_safe(row):
        params = {
            "reqid": row["Yclid"],
            "yandex-goal-id": row["GoalId"],
            "event-time": row["Timestamp"],
            "event_type": "event",
            "event": "EVENT_1",
            "pass-phrase": postback_config.PassPhrase,
            "__crypta_source__": "s2s",
        }

        if row["Value"] is not None and row["Currency"] is not None:
            params["revenue"] = row["Value"]
            params["currency"] = row["Currency"]

        try:
            return request_unsafe(params)
        except Exception as e:
            return e

    rows = list(rows)
    with cf.ThreadPoolExecutor(max_workers=postback_config.Workers) as executor:
        for row, result in zip(rows, executor.map(request_safe, rows)):
            if isinstance(result, requests.HTTPError):
                row["ResponseStatusCode"] = result.response.status_code
                row["ResponseBody"] = result.response.text
                yield row
            elif isinstance(result, Exception):
                row["Exception"] = str(result)
                yield row
