import datetime

import yaml
from yt.wrapper import ypath

from crypta.lib.python import (
    crypta_env,
    templater,
)
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import deploy
from crypta.lib.python.spine.consts import (
    environment,
    dc as dc_consts,
)
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
import crypta.lib.python.spine.deploy.consts as yp_consts
from crypta.lib.python.spine.juggler import juggler_check_generator
from crypta.lib.python.spine.juggler.aggregate_hosts import AggregateHosts
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.sandbox import (
    configs,
    sandbox_scheduler,
)
from crypta.lib.python.spine.solomon import (
    solomon_check_generator,
)
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from crypta.s2s.lib import stats_helpers
from sandbox.projects.crypta.s2s import (
    calc_stats,
    process_conversions,
    transfer_conversions_to_yt,
    upload_to_postback,
)


COMMON_AGGR_HOSTS = AggregateHosts(testing_host="crypta-s2s-testing", production_host="crypta-s2s")
ENV_TO_SANDBOX_ENV = {
    environment.TESTING: environment.TESTING,
    environment.PRODUCTION: environment.STABLE,
    environment.STABLE: environment.STABLE,
}
ENV_TO_CRYPTA_ENV_CLASS = {
    environment.TESTING: crypta_env.Testing,
    environment.PRODUCTION: crypta_env.Production,
    environment.STABLE: crypta_env.Production,
}


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-s2s"])

    add_postback(juggler)
    add_conversions_center(juggler)

    return juggler


def add_postback(base_juggler):
    juggler = base_juggler.clone(
        tags=["crypta-s2s-postback"],
        host=COMMON_AGGR_HOSTS.production_host.host,
    )

    add_common_sandbox_tasks(juggler)

    for item in get_index():
        client = item["client"]
        client_juggler = juggler.clone(tags=["crypta-s2s-{}".format(client)])

        add_per_client_sandbox_tasks(client_juggler, client)
        add_per_client_yt_metrics(client_juggler, client)

        conversion_name_to_goal_ids = stats_helpers.get_conversion_name_to_goal_ids(item)
        add_per_client_solomon_metrics(client_juggler, client, conversion_name_to_goal_ids)

    return juggler


def add_per_client_yt_metrics(juggler, client):
    yt_registry = yt_config_registry.CryptaYtConfigRegistry(juggler)
    envs = (environment.PRODUCTION, )

    for suffix, threshold in [
        ("errors/parsing", parsing_errors_threshold(client)),
        ("errors/to_postback", DataSize(b=0)),
    ]:
        YtSizeMetric(yt_registry, render_cypress_path(client, suffix), envs=envs).add_disk_space_alert(
            predicate=alert_pb2.GT,
            threshold=threshold,
        )

    YtSizeMetric(yt_registry, render_cypress_path(client, "State"), envs=envs)


def add_per_client_sandbox_tasks(base_juggler, client):
    juggler = base_juggler.clone(yt_dependencies=YtProxy.Group.offline)
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["S2S", "S2S-POSTBACK", client.upper()])
    sandbox_tasks = [
        configs.TaskConfig(
            transfer_conversions_to_yt.CryptaS2sTransferConversionsToYtTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(days=1, hours=1),
                kill_timeout=datetime.timedelta(hours=1),
                schedule_interval=datetime.timedelta(hours=6),
            ),
        ),
        configs.TaskConfig(
            process_conversions.CryptaS2sProcessConversionsTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(days=1, hours=1),
                kill_timeout=datetime.timedelta(hours=1),
                schedule_interval=datetime.timedelta(hours=1),
            ),
        ),
        configs.TaskConfig(
            upload_to_postback.CryptaS2sUploadToPostbackTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(days=1, hours=1),
                kill_timeout=datetime.timedelta(hours=1),
                schedule_interval=datetime.timedelta(hours=1),
            ),
        ),
    ]

    for task_config in sandbox_tasks:
        for config in task_config:
            config.extra_params = {"client": client}
            scheduler = sandbox.create_scheduler(**config.get_scheduler_config())
            scheduler.check(config.crit_time, juggler_service=config.task_cls.get_juggler_service_for_client(client))


def add_common_sandbox_tasks(base_juggler):
    juggler = base_juggler.clone(yt_dependencies=YtProxy.Group.offline)
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["S2S", "S2S-POSTBACK", "S2S-COMMON"])
    sandbox_tasks = [
        configs.TaskConfig(
            calc_stats.CryptaS2sCalcStatsTask,
            stable=configs.RunConfig(
                crit_time=datetime.timedelta(days=1, hours=1),
                kill_timeout=datetime.timedelta(hours=4),
                schedule_interval=datetime.timedelta(hours=6),
            ),
        ),
    ]

    for task_config in sandbox_tasks:
        for config in task_config:
            scheduler = sandbox.create_scheduler(**config.get_scheduler_config())
            scheduler.check(config.crit_time, juggler_service=config.task_cls.get_class_juggler_service())


def add_per_client_solomon_metrics(juggler, client, conversion_name_to_goal_ids):
    project = "crypta_dmp"
    cluster = environment.PRODUCTION
    service = "s2s_metrics"
    solomon = solomon_check_generator.ServiceSolomonCheckGenerator(juggler, project, service, cluster)

    solomon.create_expression_alert_check(
        service="{}.postclick-postback-ratio".format(client),
        period=datetime.timedelta(days=4),
        program="""
        let mobile_postclicks = series_sum({{
            project="{project}",
            cluster="{cluster}",
            service="{service}",
            sensor="mobile_postclick",
            client="{client}",
            conversion_name="*",
            goal_id="*"
        }});
        let uniform_postbacks = series_sum({{
            project="{project}",
            cluster="{cluster}",
            service="{service}",
            sensor="uniform_postback",
            client="{client}",
            conversion_name="*",
            goal_id="*"
        }});

        let ratio = avg(mobile_postclicks)/avg(uniform_postbacks);
        let threshold = 0.75;

        alarm_if(ratio < threshold);
        """.format(
            project=project,
            service=service,
            cluster=cluster,
            client=client,
        ),
        description="mobile_postclick/uniform_postback = {{ expression.ratio }} < {{ expression.threshold }}",
    )


def get_index():
    return yaml.safe_load(templater.render_resource("/s2s_index", vars={"environment": environment.PRODUCTION}, strict=True))


def render_cypress_path(client, suffix):
    return ypath.ypath_join("s2s/clients", client, suffix)


def parsing_errors_threshold(client):
    custom = {
        "i66": DataSize(kb=100),
    }
    return custom.get(client, DataSize(b=0))


def add_conversions_center(base_juggler):
    juggler = base_juggler.clone(tags=["crypta-s2s-conversions-center"], yt_dependencies=YtProxy.Group.offline)

    add_scheduler(juggler)
    add_conversions_downloader(juggler)
    add_conversions_downloader_sandbox(juggler)
    add_conversions_processor(juggler)


def add_scheduler(juggler):
    for aggr_host in COMMON_AGGR_HOSTS:
        crypta_env_class = ENV_TO_CRYPTA_ENV_CLASS[aggr_host.environment]
        sandbox = sandbox_scheduler.create_default_generator(juggler, ["S2S", "S2S-CONVERSIONS-CENTER"])
        sandbox.create_run_universal_bundle_scheduler(
            bundle_name="s2s-scheduler",
            cmd=["{{cwd}}/crypta-s2s-scheduler", "--config", "{{cwd}}/config.yaml"],
            env=aggr_host.environment,
            schedule_interval=datetime.timedelta(hours=1),
            kill_timeout=datetime.timedelta(hours=1),
            secrets_env=crypta_env.to_sandbox_secrets_env(
                crypta_env_class.s2s_scheduler_tvm_secret,
                crypta_env_class.grut_token,
            ),
            requirements={
                "client_tags": "SAS",
            },
        ).check(
            crit_time=datetime.timedelta(hours=4),
        ).set_host(
            aggr_host.host,
        )


def add_conversions_downloader(base_juggler):
    deploy_gen = base_juggler.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage_name = "crypta-s2s-conversions-downloader"
    stage = deploy_gen.stage(stage_name)

    for env in environment.ALL:
        crypta_env_class = ENV_TO_CRYPTA_ENV_CLASS[env]

        deploy_unit = stage.multi_cluster_deploy_unit(
            pods_per_cluster={dc_consts.SAS: 2},
            cpu_ms=500,
            ram=DataSize(gb=1),
            storage=deploy.HDD(
                capacity=DataSize(gb=5),
                bandwidth_guarantee=DataSize(mb=10),
                bandwidth_limit=DataSize(mb=10),
            ),
            project_network=yp_consts.CRYPTANETS,
            network_bandwidth=DataSize(mb=1),
            id_=env,
        ).deployment_strategy(
            max_unavailable=1,
        )

        box = deploy_unit.box(
            docker_image="crypta/crypta-s2s-conversions-downloader",
        ).docker_release_rule(
            ENV_TO_SANDBOX_ENV[env],
        ).add_init_command(
            deploy.Command(
                "/root/init.sh"
            ),
        ).literal_env(
            crypta_env_class.crypta_environment
        ).secret_env(
            crypta_env_class.yt_token
        ).secret_env(
            crypta_env_class.s2s_conversions_downloader_tvm_secret
        ).secret_env(
            crypta_env_class.grut_token
        ).secret_env(
            crypta_env_class.s2s_google_api_key
        ).secret_env(
            crypta_env_class.s2s_direct_encryption_secret
        )

        box.workload(
            start_cmd="/root/service.sh",
            id_="workload",
        )

        juggler = deploy_unit.get_juggler_generator(base_juggler.clone(
            warn_limit=0,
            crit_limit=100,
            escalation=False,
        ))

        flap_detector_params = FlapDetectorParams(datetime.timedelta(minutes=5), datetime.timedelta(minutes=10))
        juggler.icmpping().add_flap_detector(flap_detector_params)


def add_conversions_downloader_sandbox(juggler):
    for aggr_host in COMMON_AGGR_HOSTS:
        crypta_env_class = ENV_TO_CRYPTA_ENV_CLASS[aggr_host.environment]
        sandbox = sandbox_scheduler.create_default_generator(juggler, ["S2S", "S2S-CONVERSIONS-CENTER"])
        sandbox.create_run_universal_bundle_scheduler(
            bundle_name="s2s-conversions-downloader",
            cmd=["{{cwd}}/crypta-s2s-conversions-downloader-sandbox", "--config", "{{cwd}}/config.yaml"],
            env=aggr_host.environment,
            schedule_interval=datetime.timedelta(hours=1),
            kill_timeout=datetime.timedelta(hours=1),
            secrets_env=crypta_env.to_sandbox_secrets_env(
                crypta_env_class.grut_token,
                crypta_env_class.s2s_conversions_downloader_tvm_secret,
                crypta_env_class.s2s_direct_encryption_secret,
                crypta_env_class.s2s_google_api_key,
                crypta_env_class.yt_token,
                crypta_env.solomon_token,
            ),
            requirements={
                "dns": "dns64",
                "client_tags": "SAS",
            },
        ).check(
            crit_time=datetime.timedelta(hours=4),
        ).set_host(
            aggr_host.host,
        )

    project = "crypta_s2s"
    cluster = environment.PRODUCTION
    service = "conversions_downloader"
    solomon = solomon_check_generator.ServiceSolomonCheckGenerator(juggler, project, service, cluster)

    solomon.create_sensor_aggregation_check(
        service="downloader_errors",
        period=datetime.timedelta(days=1),
        labels={
            "project": project,
            "cluster": cluster,
            "sensor": "download_errors",
            "service": service,
            "client_id": "*",
        },
        predicate=alert_pb2.GT,
        threshold=0,
        description="Downloader errors: {{ expression.total }} > {{ expression.threshold }}",
    )


def add_conversions_processor(base_juggler):
    deploy_gen = base_juggler.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage_name = "crypta-s2s-conversions-processor"
    stage = deploy_gen.stage(stage_name)

    for env in environment.ALL:
        crypta_env_class = ENV_TO_CRYPTA_ENV_CLASS[env]

        deploy_unit = stage.multi_cluster_deploy_unit(
            pods_per_cluster={dc_consts.SAS: 2},
            cpu_ms=500,
            ram=DataSize(gb=1),
            storage=deploy.HDD(
                capacity=DataSize(gb=5),
                bandwidth_guarantee=DataSize(mb=10),
                bandwidth_limit=DataSize(mb=10),
            ),
            project_network=yp_consts.CRYPTANETS,
            network_bandwidth=DataSize(mb=1),
            id_=env,
        ).deployment_strategy(
            max_unavailable=1,
        )

        box = deploy_unit.box(
            docker_image="crypta/crypta-s2s-conversions-processor",
        ).docker_release_rule(
            ENV_TO_SANDBOX_ENV[env],
        ).add_init_command(
            deploy.Command(
                "/root/init.sh"
            ),
        ).literal_env(
            crypta_env_class.crypta_environment
        ).secret_env(
            crypta_env_class.yt_token
        ).secret_env(
            crypta_env_class.s2s_conversions_processor_tvm_secret
        ).secret_env(
            crypta_env_class.grut_token
        )

        box.workload(
            start_cmd="/root/service.sh",
            id_="workload",
        )

        juggler = deploy_unit.get_juggler_generator(base_juggler.clone(
            warn_limit=0,
            crit_limit=100,
            escalation=False,
        ))

        flap_detector_params = FlapDetectorParams(datetime.timedelta(minutes=5), datetime.timedelta(minutes=10))
        juggler.icmpping().add_flap_detector(flap_detector_params)
