import argparse
import logging

import tvmauth

from crypta.s2s.services.conversions_processor.lib.processor.cdp_api_client import CdpApiClient
from crypta.s2s.tools.lib.environment import Environment
from crypta.s2s.tools.lib.vault import Vault


logger = logging.getLogger(__name__)

CONVERSIONS_PROCESSOR_TVM_ID = {
    Environment.testing: 2033979,
    Environment.production: 2033977,
}
CDP_API_TVM_ID = {
    Environment.testing: 2019281,
    Environment.production: 2019283,
}
CDP_API_URL = {
    Environment.testing: "https://cdp-intapi.test.metrika.yandex.net",
    Environment.production: "https://cdp-intapi.metrika.yandex.net",
}


def add_args_subparser(subparsers):
    parser = subparsers.add_parser(
        "upload-to-crm-api",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description=(
            "Upload csv to crm api"
            "Examples:\n"
            "  crypta-s2s-tools upload-to-crm-api --environment testing --counter-id <...> --access-uid <...> --file-path <...>\n"
        )
    )

    parser.set_defaults(function=main)

    parser.add_argument("--environment", choices=Environment.all(), required=True)
    parser.add_argument("--counter-id", help="Id of counter", required=True)
    parser.add_argument("--access-uid", help="Puid of user who can write to counter", required=True)
    parser.add_argument("--file-path", help="Csv file to upload", required=True)


def main(
    environment,
    counter_id,
    access_uid,
    file_path,
):
    tvm_settings = tvmauth.TvmApiClientSettings(
        self_tvm_id=CONVERSIONS_PROCESSOR_TVM_ID[environment],
        self_secret=Vault().get_conversions_processor_tvm_secret(environment),
        dsts={
            CdpApiClient.tvm_alias: CDP_API_TVM_ID[environment],
        },
    )
    tvm_client = tvmauth.TvmClient(tvm_settings)
    cdp_api_client = CdpApiClient(CDP_API_URL[environment], tvm_client, lines_per_file=100000, logger=logger)
    cdp_api_client.simple_orders(filepath=file_path, counter_id=counter_id, uid=access_uid, metrics=None)

    return 0
