package ru.yandex.crypta.search;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.ws.rs.core.SecurityContext;

import io.grpc.stub.StreamObserver;
import org.glassfish.jersey.process.internal.RequestContext;
import org.glassfish.jersey.process.internal.RequestScope;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.solomon.Solomon;
import ru.yandex.crypta.search.cache.SearchCacheService;
import ru.yandex.crypta.search.history.UserHistoryService;
import ru.yandex.crypta.search.proto.ISearchServiceGrpc;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.crypta.search.util.CollectYield;
import ru.yandex.crypta.search.util.Metrics;
import ru.yandex.crypta.search.util.RolesHelpers;
import ru.yandex.crypta.search.util.Timer;
import ru.yandex.monlib.metrics.JvmThreads;

public class DefaultSearchService extends ISearchServiceGrpc.ISearchServiceImplBase implements SearchService {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultSearchService.class);
    public static final int DEFENSIVE_TIMEOUT_SECONDS = 120;

    private final MatchersRepository matchersRepository;
    private final Provider<SecurityContext> securityContextProvider;
    private final SearchCacheService cacheService;
    private final UserHistoryService userHistoryService;
    private final ExecutorService matchersExecutorService;

    @Inject
    private RequestScope requestScope;

    @Inject
    public DefaultSearchService(MatchersRepository matchersRepository,
                                Provider<SecurityContext> securityContextProvider,
                                SearchCacheService cacheService,
                                UserHistoryService userHistoryService) {
        this.matchersRepository = matchersRepository;
        this.securityContextProvider = securityContextProvider;
        this.cacheService = cacheService;
        this.userHistoryService = userHistoryService;
        this.matchersExecutorService = Executors.newCachedThreadPool();
        JvmThreads.addExecutorMetrics("SearchMatchers", matchersExecutorService, Solomon.REGISTRY);
    }

    @Override
    public void search(Service.TSearchRequest request, StreamObserver<Search.TResultResponse> responseObserver) {
        var result = search(request, Language.en());
        responseObserver.onNext(result);
        responseObserver.onCompleted();
    }

    @Override
    @SuppressWarnings("StreamToIterable")
    public Search.TResultResponse search(Service.TSearchRequest request, Language languageRaw) {
        userHistoryService.addSearchRequest(
                securityContextProvider.get().getUserPrincipal().getName(),
                request,
                Instant.now().getEpochSecond()
        );

        String onlyMatcher = request.getArgumentsMap().get("matcher");
        Language language = Language.orDefault(languageRaw);

        var results = createResultsList();
        Matcher.Yield<Search.TResponse.Builder> yield = results::add;

        // sync execution
        List<Matcher> matchersToRun = matchersRepository.getMatchers().stream()
                .filter(matcher -> Objects.isNull(onlyMatcher) || Objects.equals(onlyMatcher, matcher.getName()))
                .filter(matcher -> matcher.matches(request))
                .filter(matcher -> checkRole(matcher, language, request, yield))
                .collect(Collectors.toList());

        matchersToRun.forEach(matcher ->
                matchersRepository.getStats().increment(matcher.getName())
        );

        // check cache
        List<Callable<Boolean>> asyncMatcherTasks = new ArrayList<>();
        for (Matcher matcher : matchersToRun) {
            boolean isCached = processCached(matcher, request, yield);
            if (!isCached) {
                asyncMatcherTasks.add(
                        asyncMatcherCall(matcher, language, request, yield)
                );
            }
        }

        // async execution
        try {

            List<Future<Boolean>> tasks = matchersExecutorService.invokeAll(
                    asyncMatcherTasks,
                    DEFENSIVE_TIMEOUT_SECONDS, TimeUnit.SECONDS
            );

            long okExecutions = tasks.stream().map(task -> {
                try {
                    return task.get();
                } catch (InterruptedException e) {
                    LOG.error("Interrupted task: " + task);
                } catch (ExecutionException e) {
                    LOG.error("Can't execute task: " + task);
                } catch (CancellationException e) {
                    LOG.error("The task was cancelled: " + task);
                }
                return false;
            }).collect(Collectors.summarizingInt(ok -> ok ? 1 : 0)).getSum();

            LOG.info("{} out of {} matchers were executed, {} successfully",
                    asyncMatcherTasks.size(), matchersToRun.size(), okExecutions);

        } catch (InterruptedException e) {
            throw Exceptions.unchecked(e);
        }

        results.sort(Ranking.createComparator());
        var finalResults = results.stream().map(Search.TResponse.Builder::build);
        return Search.TResultResponse.newBuilder().addAllResponses(finalResults::iterator).build();
    }

    private boolean processCached(Matcher matcher,
                                  Service.TSearchRequest request,
                                  Matcher.Yield<Search.TResponse.Builder> yield) {
        Search.TResultResponse cached = null;
        if (!request.getNoCache()) {
            cached = cacheService.getCached(matcher.getName(), request);
            if (cached != null) {
                cached.getResponsesList().forEach(response -> {
                    Search.TResponse.Builder builder = response.toBuilder();
                    builder.getMetaBuilder().setCached(true);
                    yield.yield(builder);
                });
            }
        }
        return cached != null;
    }


    private Callable<Boolean> asyncMatcherCall(Matcher matcher, Language language,
                                               Service.TSearchRequest request,
                                               Matcher.Yield<Search.TResponse.Builder> yield) {

        RequestContext mainRequestContext = requestScope.current();
        Map<String, String> logContext = MDC.getCopyOfContextMap();

        return () -> {
            // allow child threads to access request scope
            // TODO: make common MDC-aware executor service
            MDC.setContextMap(logContext);
            return requestScope.runInScope(mainRequestContext, () ->
                    processMatcher(matcher, language, request, yield)
            );
        };


    }

    private List<Search.TResponse.Builder> createResultsList() {
        return Collections.synchronizedList(new ArrayList<>());
    }

    @Override
    public List<String> examples() {
        var yield = new CollectYield<List<String>, String>(ArrayList::new, example -> true);
        matchersRepository.getMatchers().forEach(each -> each.examples(yield));
        return yield.getCollected();
    }

    private void fillMeta(Search.TResponse.Builder response, Matcher matcher, Timer timer) {
        response.getMetaBuilder()
                .setMethod(matcher.getName())
                .setTimestamp(Instant.now().getEpochSecond())
                .setDuration(timer.sinceStart().toMillis());
    }

    private Search.TResponse.Builder createMissedRolesResponse(Set<String> roles, Language language) {
        var response = Search.TResponse.newBuilder();
        response.setSource("IDM")
                .getValueBuilder()
                .getMissedRolesBuilder()
                .addAllRoles(roles
                        .stream()
                        .map(each -> RolesHelpers.roleIdToPrettyName(each, language))
                        .collect(Collectors.toList()));
        return response;
    }

    private Search.TResponse.Builder createErrorResponse(String message, Matcher matcher, Timer timer) {
        var response = Search.TResponse.newBuilder();
        response.setSource("Search")
                .getValueBuilder()
                .getErrorBuilder()
                .setMessage(message);
        fillMeta(response, matcher, timer);
        return response;
    }

    private boolean processMatcher(Matcher matcher, Language language, Service.TSearchRequest request,
                                   Matcher.Yield<Search.TResponse.Builder> yield) {

        return Timer.withTimer(timer -> {
            try {
                var toCache = Search.TResultResponse.newBuilder();
                matcher.process(request, new Matcher.Context(language), value -> {
                    Metrics.processingTime(matcher).record(timer.sinceStart().toMillis());

                    var response = value.toBuilder();
                    fillMeta(response, matcher, timer);

                    toCache.addResponses(response);

                    yield.yield(response);
                });

                Duration cacheTtl = matcher.getCacheTtl();
                if (!cacheTtl.isZero() && !request.getNoCache()) {
                    cacheService.store(matcher.getName(), request, toCache.build(), cacheTtl);
                }

            } catch (Throwable t) {
                LOG.warn("Failed to process query '{}': {}", request.getQuery(), t);
                yield.yield(createErrorResponse(getMessageOrDefault(t), matcher, timer));
                return false;
            }
            return true;
        });

    }

    private boolean checkRole(Matcher matcher, Language language, Service.TSearchRequest request,
                              Matcher.Yield<Search.TResponse.Builder> yield) {
        SecurityContext securityContext = securityContextProvider.get();

        return Timer.withTimer(timer -> {
            try {
                var missedRoles = collectMissedRoles(request, matcher, securityContext);

                if (!missedRoles.isEmpty()) {
                    var response = createMissedRolesResponse(missedRoles, language);
                    fillMeta(response, matcher, timer);
                    yield.yield(response);
                    return false;
                }
            } catch (Throwable t) {
                LOG.warn("Failed to check roles for query '{}': {}", request.getQuery(), t);
                yield.yield(createErrorResponse(getMessageOrDefault(t), matcher, timer));
                return false;
            }
            return true;
        });
    }

    private String getMessageOrDefault(Throwable t) {
        return Optional.ofNullable(t.getMessage()).orElse("No message");
    }

    private Set<String> collectMissedRoles(Service.TSearchRequest request, Matcher matcher,
                                           SecurityContext securityContext) {
        var yield = new CollectYield<Set<String>, String>(
                () -> new HashSet<>(1),
                Predicate.not(securityContext::isUserInRole)
        );
        matcher.roles(request, yield);
        return yield.getCollected();
    }

}
