package ru.yandex.crypta.search;

import java.lang.reflect.Modifier;
import java.time.Duration;
import java.util.List;
import java.util.stream.Collectors;

import org.glassfish.jersey.internal.inject.InjectionManager;
import org.reflections.Reflections;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.crypta.search.util.ListYield;

public interface Matcher {

    static List<Matcher> createAllMatchers(InjectionManager injector) {
        return getReflections()
                .getSubTypesOf(Matcher.class)
                .stream()
                .filter(cls -> !Modifier.isAbstract(cls.getModifiers()))
                .filter(cls -> !Modifier.isInterface(cls.getModifiers()))
                .map(injector::createAndInitialize)
                .collect(Collectors.toList());
    }

    static Reflections getReflections() {
        return new Reflections(SearchRoot.class.getPackage().getName());
    }

    default Search.TOneofResponses.Builder createResponseValue() {
        return Search.TOneofResponses.newBuilder();
    }

    default Search.TResponse.Builder createResponse() {
        return Search.TResponse.newBuilder();
    }

    boolean matches(Service.TSearchRequest request);

    void examples(Matcher.Yield<String> yield);

    void roles(Service.TSearchRequest request, Yield<String> yield);

    void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield);

    default String getName() {
        return getClass().getSimpleName().replace("Matcher", "");
    }

    default Duration getCacheTtl() {
        return Duration.ofMinutes(1);
    }

    default Search.TMatcherDescription createDescription(Language language, String description) {

        List<String> examples = ListYield.extract(this::examples);

        ListYield<String> rolesExtractor = new ListYield<>();
        roles(Service.TSearchRequest.newBuilder().build(), rolesExtractor);

        return Search.TMatcherDescription.newBuilder()
                .setText(description)
                .setLang(language.toString())
                .addAllExamples(examples)
                .addAllRoles(rolesExtractor.getCollected())
                .build();
    }

    void description(Matcher.Yield<Search.TMatcherDescription> yield);

    interface Yield<T> {
        void yield(T response);
    }


    class Context {

        private final Language language;

        public Context(Language language) {
            this.language = language;
        }

        public Language getLanguage() {
            return language;
        }

    }

}
