package ru.yandex.crypta.search.audience;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.inject.Inject;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.audience.AudienceService;
import ru.yandex.crypta.audience.proto.TSegmentState;
import ru.yandex.crypta.audience.proto.TUserDataStats;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.lang.number.UnsignedLong;

public class AudienceMatcher implements RegexMatcher {

    private static final YPath SEGMENTS_SIMPLE =
            YPath.simple("//home/audience/production/export/segments_simple");

    @Inject
    private AudienceService audience;

    @Inject
    private LabService lab;

    @Inject
    private YtService yt;

    @Override
    public Pattern regex() {
        return Pattern.compile("^audience (?<id>[0-9]+)$", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("audience 123456");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }
        var id = matcher.group("id");
        getMetadataResponse(id).ifPresent(yield::yield);
        getStatsResponse(id, context.getLanguage()).ifPresent(yield::yield);
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Describe Audience segment"
        ));
    }

    private Optional<Search.TResponse> getMetadataResponse(String id) {
        var response = createResponse();
        response.setSource("yt");

        var audience = response.getValueBuilder().getAudienceBuilder();
        getState(id)
                .ifPresent(audience::setState);
        getParameters(id)
                .ifPresent(audience::putAllParameters);

        return Optional.of(response.build());
    }

    private Optional<Search.TResponse> getStatsResponse(String id, Language language) {
        var response = createResponse();
        response.setSource("yt");

        var value = response.getValueBuilder().getUserDataStatsBuilder();
        value.setId(id);
        getStats(id)
                .map(getLab(language)::getSimpleSampleStats)
                .ifPresent(value::setStats);

        return Optional.of(response.build());
    }

    private Optional<Map<String, String>> getParameters(String id) {
        var ulId = UnsignedLong.valueOf(Long.parseUnsignedLong(id));
        var pathWithKey = SEGMENTS_SIMPLE.withExact(YtReadingUtils.exact(ulId));

        ListF<Map<String, String>> records =
                yt.readTableYson(pathWithKey, each -> Cf.wrap(each.asMap()).mapValues(value -> {
                    if (value.isStringNode()) {
                        return value.stringValue();
                    }
                    if (value.isIntegerNode()) {
                        return Long.toUnsignedString(value.longValue());
                    }
                    if (value.isEntityNode()) {
                        return "<null>";
                    }
                    return "<unsupported>";
                }));

        return records.firstO().toOptional();
    }

    private Optional<TUserDataStats> getStats(String id) {
        return tryTypes(Set.of("audience", "lookalike_output"), type -> audience.getStats(type, id));
    }

    private Optional<TSegmentState> getState(String id) {
        return tryTypes(Set.of("regular", "lookalike"), type -> audience.getState(type, id));
    }

    private LabService getLab(Language language) {
        return lab.withLanguage(Language.orDefault(language));
    }

    private <T> Optional<T> tryTypes(Set<String> types, Function<String, T> getByType) {
        return types
                .stream()
                .flatMap(type -> {
                    try {
                        return Optional.of(getByType.apply(type)).stream();
                    } catch (Exception ex) {
                        return Optional.<T>empty().stream();
                    }
                }).findFirst();
    }
}
