package ru.yandex.crypta.search.cache;

import java.time.Duration;

import javax.inject.Inject;

import com.google.protobuf.InvalidProtocolBufferException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.redis.RedisClient;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.misc.digest.Sha256;

public class RedisSearchCacheService implements SearchCacheService {

    private static final Logger LOG = LoggerFactory.getLogger(RedisSearchCacheService.class);

    private final RedisClient redis;

    @Inject
    public RedisSearchCacheService(RedisClient redis) {
        this.redis = redis;
    }

    @Override
    public void store(String matcher, Service.TSearchRequest request, Search.TResultResponse response, Duration ttl) {
        String requestKey = Sha256.A.digest(request.toByteArray()).hex();
        try {
            redis.setBytes(getClass(), matcher, requestKey, response.toByteArray(), ttl);
            LOG.info(
                    "Stored {} items for {}({}) by key {}",
                    response.getResponsesCount(), matcher, request, requestKey
            );
        } catch (Exception e) {
            LOG.warn("Can't store result to a cache for {}({}) by key {} ", matcher, request, requestKey);
        }
    }

    @Override
    public Search.TResultResponse getCached(String matcher, Service.TSearchRequest request) {
        String requestKey = Sha256.A.digest(request.toByteArray()).hex();
        try {
            byte[] bytes = redis.getBytes(getClass(), matcher, requestKey);
            if (bytes == null) {
                LOG.info("No cached result for {}({}) at key {}", matcher, request, requestKey);
                return null;
            }

            Search.TResultResponse result = Search.TResultResponse.parseFrom(bytes);
            LOG.info("Found {} items for {}({}) at key {}", result.getResponsesCount(), matcher, request, requestKey);
            return result;
        } catch (InvalidProtocolBufferException e) {
            LOG.warn("Can't parse cached result for {}({}) at key {}", matcher, request, requestKey);
            return null;
        } catch (Exception e) {
            LOG.warn("Can't fetch cached result for {}({}) at key {}", matcher, request, requestKey);
            return null;
        }
    }
}
