package ru.yandex.crypta.search.caesar;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import org.jetbrains.annotations.NotNull;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.impl.ytree.serialization.YTreeTextSerializer;
import ru.yandex.ysonjsonconverter.YsonJsonConverter;

public class CaesarMatcher implements RegexMatcher {

    private static final Pattern CAESAR_PATTERN =
            Pattern.compile("(caesar )((?<entity>\\w+) )((?<idKey>\\w+)=(?<idValue>\\d+))");
    private static final String CAESAR_LOOKUP_URL = "https://lookup-profile-caesar1.n.yandex-team.ru";

    private final OkHttpClient client;

    public CaesarMatcher() {
        this.client = new OkHttpClient.Builder()
                .readTimeout(5, TimeUnit.SECONDS)
                .build();
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("caesar Banners BannerID=123");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
        // TODO
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }
        var entity = matcher.group("entity");
        var idKey = matcher.group("idKey");
        var idValue = matcher.group("idValue");

        Request req = buildRequest(entity, idKey, idValue);
        try (var response = client.newCall(req).execute()) {
            var responseBody = Objects.requireNonNull(response.body());
            var yTreeNode = YTreeTextSerializer.deserialize(responseBody.string());
            var jsonNode = YsonJsonConverter.yson2json(JsonNodeFactory.instance, yTreeNode);
            yield.yield(createResponse()
                    .setValue(createResponseValue()
                            .setJson(jsonNode.toString())
                            .build())
                    .build());
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @NotNull
    private Request buildRequest(String entity, String idKey, String idValue) {
        var url = HttpUrl
                .parse(CAESAR_LOOKUP_URL)
                .newBuilder()
                .addPathSegment("lookup_profile")
                .addPathSegment(entity)
                .addQueryParameter(idKey, idValue)
                .build();

        return new Request.Builder()
                .url(url.toString())
                .get()
                .build();
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(Language.EN, "Fetches data from CaeSaR (https://a.yandex-team.ru/arc/trunk/arcadia/ads/bsyeti/caesar/libs/profiles/proto)"));
    }

    @Override
    public Pattern regex() {
        return CAESAR_PATTERN;
    }
}
