package ru.yandex.crypta.search.cm;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import javax.inject.Inject;

import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import retrofit2.Retrofit;
import retrofit2.converter.jackson.JacksonConverterFactory;

import ru.yandex.crypta.clients.tvm.TvmClient;
import ru.yandex.crypta.clients.tvm.TvmOkHttpInterceptor;
import ru.yandex.crypta.clients.utils.OkHttpUtils;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.proto.TCmConfig;
import ru.yandex.crypta.lib.proto.TCmTvmConfig;
import ru.yandex.crypta.search.Matcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class CmMatcher implements Matcher {

    private static final Pattern EXT_ID_PATTERN = Pattern.compile("(?<type>[^:]+):(?<value>.*)");
    private static final Pattern YANDEXUID_PATTERN = Pattern.compile("((yandexuid|yuid) )?(?<yandexuid>\\d{13,20})");
    private static final Logger LOG = LoggerFactory.getLogger(CmMatcher.class);

    private final TCmConfig config;
    private final OkHttpClient client;


    @Inject
    public CmMatcher(TCmConfig cmConfig, TCmTvmConfig cmTvmConfig, TvmClient tvm) {
        this.config = cmConfig;
        tvm.setDstClientIds(List.of(cmTvmConfig.getDestinationTvmId()));
        this.client = new OkHttpClient.Builder()
                .readTimeout(5, TimeUnit.SECONDS)
                .addInterceptor(new TvmOkHttpInterceptor(tvm, cmTvmConfig.getDestinationTvmId()))
                .build();
    }

    @Override
    public boolean matches(Service.TSearchRequest request) {
        return EXT_ID_PATTERN.matcher(request.getQuery()).matches() ||
                YANDEXUID_PATTERN.matcher(request.getQuery()).matches();
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("fpc:63d76f7d2d299ede");
        yield.yield("duid:1580608008767669748");
        yield.yield("yandexuid:6422603701518798182");
        yield.yield("6422603701518798182");
        yield.yield("yandexuid 6422603701518798182");

    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        String type;
        String value;

        var extIdMatcher = EXT_ID_PATTERN.matcher(request.getQuery());
        var yandexuidMatcher = YANDEXUID_PATTERN.matcher(request.getQuery());

        if (extIdMatcher.matches()) {
            type = extIdMatcher.group("type").strip();
            value = extIdMatcher.group("value").strip();
        } else if (yandexuidMatcher.matches()) {
            type = "yandexuid";
            value = yandexuidMatcher.group("yandexuid");
        } else {
            LOG.error("Query '{}' does not contain suitable string", request.getQuery());
            return;
        }

        try {
            var response = getCmApi().identify(type, value, "search").execute();

            if (response.code() == OkHttpUtils.NOT_FOUND) {
                return;
            }

            if (!response.isSuccessful()) {
                LOG.error("CM returned an unexpected code {} for search query = `{}`", response.code(), request.getQuery());
                return;
            }

            response.body().forEach(node -> {
                var builder = createResponseValue();
                builder.getCmBuilder()
                        .setType(node.getType())
                        .setValue(node.getValue())
                        .setMatchTs(node.getMatchTs())
                        .putAllAttributes(node.getAttributes());
                yield.yield(createResponse().setValue(builder).setSource("CM").build());
            });
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show matched cookies from Cookie Matching service"
        ));
    }

    private CmApi getCmApi() {
        var url = new HttpUrl.Builder()
                .scheme("http")
                .host(config.getHost())
                .build();
        return new Retrofit.Builder()
                .baseUrl(url)
                .addConverterFactory(JacksonConverterFactory.create())
                .client(client)
                .build()
                .create(CmApi.class);
    }

}
