package ru.yandex.crypta.search.custom_audience;

import java.text.MessageFormat;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Splitter;

import ru.yandex.crypta.audience.proto.TAge;
import ru.yandex.crypta.audience.proto.TCity;
import ru.yandex.crypta.audience.proto.TCountry;
import ru.yandex.crypta.audience.proto.TDevice;
import ru.yandex.crypta.audience.proto.TGender;
import ru.yandex.crypta.audience.proto.TIncome;
import ru.yandex.crypta.audience.proto.TSegment;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.search.util.ParamsParsing;
import ru.yandex.crypta.siberia.bin.custom_audience.common.proto.TCaRule;

public class ParamsParser {
    private static final List<ParamsParsing.FieldParser> PARSERS;

    static {
        PARSERS = List.of(
            new ParamsParsing.EnumAttributeParser("Gender", TGender.values()),
            new ParamsParsing.EnumRepeatedFieldParser("Incomes", TIncome.values(), (x)-> Splitter.on("_").splitToList(x).get(1).toLowerCase()),
            new ParamsParsing.EnumRepeatedFieldParser("Ages", TAge.values(), (x)->x.replaceFirst("FROM_", "").replace("_TO_", "-")),
            new ParamsParsing.EnumRepeatedFieldParser("Devices", TDevice.values()),
            new ParamsParsing.EnumRepeatedFieldParser("Cities", TCity.values()),
            new ParamsParsing.EnumRepeatedFieldParser("Countries", TCountry.values()),
            new ParamsParsing.RepeatedFieldParser("Regions", (rawValue) -> {
                try {
                    return Integer.parseInt(rawValue);
                } catch (NumberFormatException e) {
                    throw Exceptions.wrongRequestException(MessageFormat.format("Regions should be integer, got ''{1}''", rawValue), "BAD_REQUEST");
                }
            }, "<int>"),

            new ParamsParsing.RepeatedFieldParser("Hosts"),
            new ParamsParsing.RepeatedFieldParser("Words"),
            new ParamsParsing.RepeatedFieldParser("AffinitiveSites"),
            new ParamsParsing.RepeatedFieldParser("TopCommonSites"),
            new ParamsParsing.RepeatedFieldParser("Segments", (segment) -> {
                try {
                    var parsed = Arrays.stream(segment.split("/")).mapToInt(Integer::parseInt).toArray();
                    return TSegment.newBuilder().setKeyword(parsed[0]).setID(parsed[1]).build();
                } catch (IndexOutOfBoundsException|NumberFormatException e) {
                    throw Exceptions.wrongRequestException("Segment should be in format 'keywordID/segmentID'", "BAD_REQUEST");
                }
            }, "<keywordID/segmentID>")
        );
    }

    private static class CaRuleConsumer implements ParamsParsing.Consumer {
        private TCaRule.Builder parentBuilder;

        public CaRuleConsumer() {
            this.parentBuilder = TCaRule.newBuilder();
        }

        @Override
        public <T> void consume(String fieldName, T value) {
            var fieldDesc = parentBuilder.getDescriptorForType().findFieldByName(fieldName);
            parentBuilder.setField(fieldDesc, value);
        }

        @Override
        public <T> void consumeMultiple(String fieldName, Stream<T> values) {
            var fieldDesc = parentBuilder.getDescriptorForType().findFieldByName(fieldName);
            values.forEach(value -> parentBuilder.addRepeatedField(fieldDesc, value));
        }

        public TCaRule getCaRule() {
            var result = parentBuilder.build();
            if (requestIsEmpty(result)) {
                return null;
            }
            return result;
        }
    }

    public static TCaRule parseArgs(String rawArgs) {
        var caRuleConsumer = new CaRuleConsumer();

        ParamsParsing.parseArgs(rawArgs, PARSERS, caRuleConsumer);

        return caRuleConsumer.getCaRule();
    }

    public static String usage() {
        return MessageFormat.format(
            "Possible arguments:\n{0}",
            PARSERS.stream().map(ParamsParsing.FieldParser::usage).collect(Collectors.joining("\n"))
        );
    }

    private static Boolean requestIsEmpty(TCaRule rule) {
        return rule.toByteArray().length == 0;
    }
}
