package ru.yandex.crypta.search.fp;

import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.inject.Inject;

import org.jetbrains.annotations.NotNull;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.graph.rt.events.proto.TFingerprint;
import ru.yandex.crypta.graph.rt.fp.herschel.proto.THerschelState;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.crypta.service.bigrt.BigRTHelper;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeKeyField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeSerializerClass;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeBytesToStringSerializer;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeObjectSerializer;
import ru.yandex.yt.ytclient.proxy.MappedLookupRowsRequest;

import static ru.yandex.crypta.search.util.YabsHash.calcHalfMd5;

public class RTFPMatcher implements RegexMatcher {

    private static final YPath TABLE = YPath.simple("//home/crypta/testing/rtsklejka/state/herschel_state");
    private final YtService yt;
    private final BigRTHelper bigRTHelper;

    @Inject
    public RTFPMatcher(YtService yt) {
        this.yt = yt;
        this.bigRTHelper = new BigRTHelper();
    }

    @Override
    public Pattern regex() {
        return Pattern.compile("rtfp ip=(?<ip>[A-Za-z0-9:.]+) ua=(?<ua>.+)");
    }

    @Override
    public void examples(Yield<String> yield) {

    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {

    }

    @SuppressWarnings("UnusedVariable")
    @YTreeObject
    public static class FPRow {
        @YTreeField(key = "Fp")
        @YTreeKeyField
        @YTreeSerializerClass(YTreeBytesToStringSerializer.class)
        private byte[] fp;

        @YTreeField(key = "State")
        @YTreeSerializerClass(YTreeBytesToStringSerializer.class)
        private byte[] state;

        public FPRow(byte[] fp) {
            this.fp = fp;
        }
    }

    @NotNull
    private MappedLookupRowsRequest<FPRow> getRequest(TFingerprint fp) {
        MappedLookupRowsRequest<FPRow> request = new MappedLookupRowsRequest<>(
                TABLE.toString(),
                new YTreeObjectSerializer<>(FPRow.class)
        );
        request.addAllLookupColumns();
        request.addFilter(new FPRow(fp.toByteArray()));
        return request;
    }


    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {

        Matcher matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }

        String ipv6 = matcher.group("ip");
        String userAgent = matcher.group("ua");

        TFingerprint fpKey = TFingerprint.newBuilder()
                .setUAHash(calcHalfMd5(userAgent.getBytes(StandardCharsets.UTF_8)))
                .setUserIP(ipv6)
                .build();

        yt.getHahnRpc().lookupRows(getRequest(fpKey)).thenAccept(rows ->
                rows.getYTreeRows()
                        .stream()
                        .map(row ->
                                bigRTHelper.parseState(
                                        row.getBytes("State"),
                                        THerschelState.parser()
                                ))
                        .map(state -> {
                            Search.TResponse.Builder responseBuilder = Search.TResponse.newBuilder();
                            responseBuilder
                                    .getValueBuilder()
                                    .getRtfpResponseBuilder()
                                    .setKey(fpKey)
                                    .setState(state);
                            return responseBuilder.build();
                        })
                        .forEach(yield::yield)
        ).join();
    }


    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Your realtime fingerprint"
        ));
    }
}
