package ru.yandex.crypta.search.help;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;
import javax.inject.Provider;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.search.Matcher;
import ru.yandex.crypta.search.MatchersRepository;
import ru.yandex.crypta.search.id.HumanishIdMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.crypta.search.util.MapYield;

public class HelpMatcher implements Matcher {

    private static final Logger LOG = LoggerFactory.getLogger(HelpMatcher.class);

    private static final Comparator<Search.TMatcherDescription> MATCHERS_USAGE_COMPARATOR = Comparator
            .comparingLong(Search.TMatcherDescription::getUsageCount)
            .reversed();
    private final Provider<MatchersRepository> matchersRepositoryProvider;

    @Inject
    public HelpMatcher(Provider<MatchersRepository> matchersRepositoryProvider) {
        this.matchersRepositoryProvider = matchersRepositoryProvider;
    }

    @Override
    public boolean matches(Service.TSearchRequest request) {
        return "help".equals(request.getQuery()) || "/help".equals(request.getQuery());
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("help");
        yield.yield("/help");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    private String ruEngString(String ru, String eng, Language lang) {
        if (lang.isRu()) {
            return ru;
        }
        return eng;
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        Language lang = context.getLanguage();
        String enLangCode = Language.EN.toString();

        var likeIdDescription = new ArrayList<Search.TMatcherDescription>();
        var otherDescription = new ArrayList<Search.TMatcherDescription>();

        MatchersRepository matchersRepository = matchersRepositoryProvider.get();
        for (Matcher matcher : matchersRepository.getMatchers()) {
            // get one per language
            Map<String, Search.TMatcherDescription> descriptions = MapYield.extract(
                    matcher::description,
                    Search.TMatcherDescription::getLang
            );

            // get proper language or ENG
            Search.TMatcherDescription desc = descriptions.getOrDefault(
                    lang.toString(),
                    descriptions.get(enLangCode)
            );

            if (desc == null) {
                LOG.warn("Matcher {} doesn't have a description", matcher.getName());
                desc = Search.TMatcherDescription.newBuilder()
                        .setName(matcher.getName())
                        .build();
            }

            desc = desc.toBuilder().setUsageCount(
                    matchersRepository.getStats().getUsageCount(matcher.getName())
            ).build();

            if (matcher instanceof HumanishIdMatcher) {
                // combine similar matchers later
                likeIdDescription.add(desc);
            } else {
                otherDescription.add(desc);
            }
        }

        Search.TMatcherDescription likeIdCombinedMatcherDesc = combineDescriptions(
                likeIdDescription,
                "IdInfoMatcher",
                ruEngString("Информация об идентификаторе", "Show id info", lang),
                lang
        );

        Stream.concat(
                Stream.of(likeIdCombinedMatcherDesc),
                otherDescription.stream()
        ).sorted(
                MATCHERS_USAGE_COMPARATOR
        ).forEach(
                d -> yieldHelpResponse(yield, d, lang)
        );

    }

    private Search.TMatcherDescription combineDescriptions(List<Search.TMatcherDescription> descriptions,
                                                           String commonName,
                                                           String commonText,
                                                           Language language) {
        List<String> examples = descriptions
                .stream()
                .flatMap(desc -> desc.getExamplesList().stream())
                .collect(Collectors.toList());
        List<String> roles = descriptions
                .stream()
                .flatMap(desc -> desc.getRolesList().stream())
                .collect(Collectors.toList());

        long usageCount = descriptions
                .stream()
                .mapToLong(Search.TMatcherDescription::getUsageCount)
                .sum();

        return Search.TMatcherDescription.newBuilder()
                .setName(commonName)
                .setText(commonText)
                .setLang(language.toString())
                .setUsageCount(usageCount)
                .addAllExamples(examples)
                .addAllRoles(roles)
                .build();
    }

    private void yieldHelpResponse(Yield<Search.TResponse> yield, Search.TMatcherDescription desc, Language lang) {

        String examples = desc
                .getExamplesList()
                .stream()
                .map(example -> "* " + example)
                .collect(Collectors.joining("\n"));

        String roles = desc
                .getRolesList()
                .stream()
                .map(example -> "* " + example)
                .collect(Collectors.joining("\n"));

        String text = desc.getText();
        if (!examples.isEmpty()) {
            text += "\n\n" + ruEngString("Примеры", "Examples", lang) + ":\n" + examples;
        }
        if (!roles.isEmpty()) {
            text += "\n\n" + ruEngString("Роли", "Roles", lang) + ":\n" + roles;
        }

        long usageCount = desc.getUsageCount();
        text += "\n\n" + ruEngString("Статистика недавнего использования", "Recent usage stats", lang) + ": " + usageCount;

        yield.yield(createResponse()
                .setValue(createResponseValue().setText(text))
                .setSource(desc.getName())
                .build()
        );
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show this help"
        ));
        yield.yield(createDescription(
                Language.RU, "Показать эту справку"
        ));
    }
}
