package ru.yandex.crypta.search.history;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import com.google.common.base.Splitter;
import com.google.protobuf.InvalidProtocolBufferException;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.redis.RedisClient;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

import static ru.yandex.crypta.search.history.UserHistoryMatcher.HISTORY_COMMAND;

public class RedisUserHistoryService implements UserHistoryService {

    private static final Logger LOG = LoggerFactory.getLogger(RedisUserHistoryService.class);

    public static final int STORE_ITEMS_COUNT = 100;
    public static final int GET_UNIQUE_ITEMS_COUNT = 40;
    public static final String SEARCH_REQUESTS_FIELD = "searchRequests";
    public static final String TOTAL_SEARCH_REQUEST_COUNT_FIELD = "totalSearchRequestCount";

    private RedisClient redisClient;

    @Inject
    public RedisUserHistoryService(RedisClient redisClient) {
        this.redisClient = redisClient;
    }


    @Override
    public List<Search.THistoricalSearchRequest> getSearchRequests(String login) {
        try {
            var items = redisClient
                    .getItems(getClass(), login, SEARCH_REQUESTS_FIELD, STORE_ITEMS_COUNT)
                    .stream().flatMap(proto -> {
                        try {
                            return Stream.of(Search.THistoricalSearchRequest.parseFrom(proto));
                        } catch (InvalidProtocolBufferException e) {
                            return Stream.of();
                        }
                    }).collect(Collectors.toList());

            // get only unique queries with last ts
            // TODO: think of using redis set ot map
            return getUniqueNQueries(items, GET_UNIQUE_ITEMS_COUNT);
        } catch (Exception e) {
            LOG.error("Can't fetch user search history for {}", login);
            return List.of();
        }
    }

    @NotNull
    private List<Search.THistoricalSearchRequest> getUniqueNQueries(List<Search.THistoricalSearchRequest> items,
                                                                    int n) {
        List<Search.THistoricalSearchRequest> result = new ArrayList<>();

        Search.THistoricalSearchRequest currentItem = null;
        for (Search.THistoricalSearchRequest item : items) {
            if (currentItem == null || !currentItem.getQuery().equals(item.getQuery())) {
                currentItem = item;

                if (result.size() < n) {
                    result.add(currentItem);
                } else {
                    break;
                }
            }
        }

        return result;
    }

    @Override
    public void addSearchRequest(String login, Search.THistoricalSearchRequest request) {
        try {
            if (!HISTORY_COMMAND.equals(request.getQuery())) {
                byte[] bytes = request.toByteArray();
                redisClient.addItem(getClass(), login, SEARCH_REQUESTS_FIELD, bytes, STORE_ITEMS_COUNT);
                redisClient.incr(getClass(), login, TOTAL_SEARCH_REQUEST_COUNT_FIELD);
            }
        } catch (Exception e) {
            LOG.error("Can't store search request to history for {}", login);
        }
    }

    @Override
    public void addSearchRequest(String login, Service.TSearchRequest searchRequest, long ts) {
        var historicalSearchRequest = Search.THistoricalSearchRequest.newBuilder()
                .setQuery(searchRequest.getQuery())
                .setTs(ts)
                .build();
        addSearchRequest(login, historicalSearchRequest);
    }

    @Override
    public Map<String, Long> getUsers() {
        return redisClient.listKeys(getClass())
                .stream()
                .map(key -> Splitter.on(":").splitToList(key).get(1)) // second key part is login
                .collect(Collectors.toSet())
                .stream()
                .collect(Collectors.toMap(
                        login -> login,
                        login -> redisClient.getLong(getClass(), login, TOTAL_SEARCH_REQUEST_COUNT_FIELD)
                ));
    }

}
