package ru.yandex.crypta.search.history;

import java.time.Duration;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.ws.rs.core.SecurityContext;

import ru.yandex.crypta.search.Matcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class UserHistoryMatcher implements Matcher {

    public static final Comparator<Search.THistoricalSearchRequest> REVERSED_TS_COMPARATOR =
            Comparator.comparingLong(Search.THistoricalSearchRequest::getTs).reversed();
    public static final String HISTORY_COMMAND = "/history";

    private final UserHistoryService userHistoryService;
    private final Provider<SecurityContext> securityContextProvider;

    @Inject
    public UserHistoryMatcher(UserHistoryService userHistoryService,
                              Provider<SecurityContext> securityContextProvider) {
        this.userHistoryService = userHistoryService;
        this.securityContextProvider = securityContextProvider;
    }

    @Override
    public boolean matches(Service.TSearchRequest request) {
        return HISTORY_COMMAND.equals(request.getQuery());
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("/history");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {

    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        String login = securityContextProvider.get().getUserPrincipal().getName();

        List<Search.THistoricalSearchRequest> requests = userHistoryService
                .getSearchRequests(login)
                .stream()
                .sorted(REVERSED_TS_COMPARATOR)
                .collect(Collectors.toList());

        for (var searchRequest : requests) {
            Search.TResponse.Builder response = createResponse();
            response.getValueBuilder().setHistoricalSearchRequest(searchRequest);

            yield.yield(response.build());
        }
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {

    }

    @Override
    public Duration getCacheTtl() {
        return Duration.ZERO;
    }
}
