package ru.yandex.crypta.search.id;

import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;

import javax.inject.Inject;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import org.jetbrains.annotations.NotNull;
import yabs.proto.Profile;

import ru.yandex.crypta.clients.bigb.BigbClient;
import ru.yandex.crypta.clients.bigb.BigbIdType;
import ru.yandex.crypta.clients.bigb.ProfileService;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.proto.Me;
import ru.yandex.crypta.search.Matcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Search.TResponse;
import ru.yandex.crypta.search.proto.Service;

public abstract class HumanishIdMatcher implements Matcher {

    @Inject
    private ProfileService profileService;

    @Inject
    private BigbClient bigbClient;

    private Pattern pattern;

    protected HumanishIdMatcher() {
        pattern = createPattern();
    }

    private Pattern createPattern() {
        var aliases = String.join("|", getIdentifierAliases());
        return Pattern.compile(String.format("(((%s) )?(?<id>.*))", aliases), Pattern.CASE_INSENSITIVE);
    }

    @Override
    public boolean matches(Service.TSearchRequest request) {
        var matcher = pattern.matcher(request.getQuery());
        if (matcher.matches()) {
            var id = matcher.group("id");
            return getIdentifier(id).isValid();
        }
        return false;
    }

    @Override
    public void examples(Yield<String> yield) {
        var next = getIdentifier("").getNext();
        var query = getIdentifierAliases().stream().findFirst().orElse("") + " " + next;
        yield.yield(query);
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    public abstract Identifier getIdentifier(String query);

    public abstract List<String> getIdentifierAliases();

    public BigbIdType getBigbIdType() {
        return BigbIdType.BIGB_UID;
    }

    protected String extractId(String query) {
        var matcher = pattern.matcher(query);
        if (matcher.matches()) {
            return matcher.group("id");
        } else {
            throw Exceptions.illegal("Query is invalid");
        }
    }

    public void yieldSimpleBigbResults(Identifier id, BigbIdType bigbIdType, Language language,
                                       Yield<TResponse> yield) {
        var profile = profileService.get(bigbIdType, id.getNormalizedValue(), language);

        TResponse socdemResponse = extractSocdem(id, profile);
        if (isProfileNonEmpty(socdemResponse.getValue().getSocdem())) {
            yield.yield(socdemResponse);
        }

        TResponse geoResponse = extractGeo(id, profile);
        Search.TGeo geo = geoResponse.getValue().getGeo();
        if (!Objects.equals(geo, geo.getDefaultInstanceForType())) {
            yield.yield(geoResponse);
        }
    }

    @NotNull
    private TResponse extractGeo(Identifier id, Me.Profile profile) {
        String geoDetailsUrl = String.format(
                "https://crypta.yandex-team.ru/geo?uid=%s&uidType=%s",
                id.getValue(),
                Soup.CONFIG.name(id.getType())
        );
        var geoData = Search.TGeo.newBuilder()
                .addAllRegular(profile.getRegularGeoList())
                .addAllActual(profile.getActualGeoList())
                .build();
        return createResponse()
                .setValue(createResponseValue().setGeo(geoData))
                .setSource("BigB")
                .setMeta(TResponse.TMeta.newBuilder().setDetailsUrl(geoDetailsUrl).build())
                .build();
    }

    @NotNull
    private TResponse extractSocdem(Identifier id, Me.Profile profile) {
        var socdemData = Search.TSocdem.newBuilder()
                .setAge(profile.getAge())
                .setGender(profile.getGender())
                .setIncome(profile.getIncome())
                .setExactSocdem(profile.getExactDemographics())
                .setExactSocdemRt(profile.getRtExactDemographics())
                .build();

        var detailsSocdemUrl = String.format(
                "https://crypta.yandex-team.ru/me?uid=%s&uidType=%s",
                id.getValue(),
                Soup.CONFIG.name(id.getType())
        );
        return createResponse()
                .setValue(createResponseValue().setSocdem(socdemData))
                .setSource("BigB")
                .setMeta(TResponse.TMeta.newBuilder().setDetailsUrl(detailsSocdemUrl))
                .build();
    }

    private void yieldRawBigbResult(Identifier id, BigbIdType bigbIdType, Yield<TResponse> yield) {
        var bigbResponse = bigbClient.getBigbDataProto(bigbIdType, id.getNormalizedValue());

        var detailsUrl = String.format(
                "https://crypta.yandex-team.ru/profiles?chapter=Other&uid=%s&uidType=%s",
                id.getValue(),
                Soup.CONFIG.name(id.getType())
        );
        var response = createResponse()
                .setValue(createResponseValue().setJson(toJson(bigbResponse)))
                .setSource("BigB raw")
                .setMeta(TResponse.TMeta.newBuilder().setDetailsUrl(detailsUrl).build())
                .build();
        yield.yield(response);
    }

    private String toJson(Profile bigbResponse) {
        try {
            return JsonFormat.printer().print(bigbResponse);
        } catch (InvalidProtocolBufferException e) {
            throw Exceptions.unchecked(e);
        }
    }

    private boolean isProfileNonEmpty(Search.TSocdem socdemData) {
        return !(Objects.equals(socdemData.getAge(), socdemData.getAge().getDefaultInstanceForType()) &&
                Objects.equals(socdemData.getGender(), socdemData.getGender().getDefaultInstanceForType()) &&
                Objects.equals(socdemData.getIncome(), socdemData.getIncome().getDefaultInstanceForType()) &&
                Objects.equals(socdemData.getExactSocdem(), socdemData.getExactSocdem().getDefaultInstanceForType()) &&
                Objects.equals(socdemData.getExactSocdemRt(), socdemData.getExactSocdemRt().getDefaultInstanceForType())
        );
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<TResponse> yield) {
        Identifier identifier = getIdentifier(extractId(request.getQuery()));
        BigbIdType bigbIdType = getBigbIdType();

        yieldSimpleBigbResults(identifier, bigbIdType, context.getLanguage(), yield);
        yieldRawBigbResult(identifier, bigbIdType, yield);
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Get identifier info"
        ));
    }
}
