package ru.yandex.crypta.search.id;

import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.lib.proto.identifiers.TIdType;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

import static java.util.stream.Collectors.joining;

public class IdentifiersMatcher implements RegexMatcher {

    private final List<TIdType> supportedIdTypes;

    public IdentifiersMatcher() {
        supportedIdTypes = Soup.CONFIG.getIdTypes().getAll().stream().filter(idType -> {
            try {
                // trying to instantiate in identifiers lib
                new Identifier(idType.getType(), "").getNext();
                return true;
            } catch (Exception ignored) {
                return false;
            }
        }).collect(Collectors.toList());

    }

    @Override
    public Pattern regex() {
        var aliases = supportedIdTypes.stream().map(TIdType::getName).collect(joining("|"));
        return Pattern.compile(String.format("(?<idType>%s) (?<id>.*)\\b", aliases), Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("yandexuid 123");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {

    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }

        String id = matcher.group("id");
        String idTypeStr = matcher.group("idType");
        TIdType idType = Soup.CONFIG.getIdType(idTypeStr);

        Identifier identifier = new Identifier(idType.getType(), id);

        var response = createResponse();

        response.getValueBuilder().getIdentifiersLibBuilder()
                .setId(id)
                .setIdType(idTypeStr)
                .setIsValid(identifier.isValid())
                .setNormalized(identifier.getNormalizedValue())
                .setMd5(identifier.getMd5())
                .setSha256(identifier.getSha256());

        response.getMetaBuilder().setDetailsUrl(
                "https://a.yandex-team.ru/arc/trunk/arcadia/yql/udfs/crypta/identifiers"
        );

        yield.yield(response
                .setSource("identifiers lib")
                .build()
        );

    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.RU, "Провалидировать и нормализовать идентификатор"
        ));
        yield.yield(createDescription(
                Language.EN, "Validate and normalize id"
        ));
    }

}
