package ru.yandex.crypta.search.id;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.api.model.graph.Graph;
import ru.yandex.crypta.graph.api.model.graph.Vertex;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.graph.api.service.YtSoupGraphService;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.idm.Roles.Lab.Matching;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.ERepresentedObjectType;
import ru.yandex.crypta.lib.proto.identifiers.TIdType;
import ru.yandex.crypta.search.Matcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

import static java.util.Map.entry;

public class SoupLinksIdMatcher implements Matcher {

    private static final List<String> NON_PRIVATE_ID_TYPES = Soup.CONFIG
            .getIdTypes()
            .getAll()
            .stream()
            .filter(idType -> idType.getRepObj() == ERepresentedObjectType.DEVICE
                    || idType.getRepObj() == ERepresentedObjectType.APP
                    || idType.getRepObj() == ERepresentedObjectType.BROWSER)
            .map(TIdType::getName)
            .collect(Collectors.toList());

    private static final Map<String, Matching.Type> PRIVATE_ID_TYPES_MAPPING = Map.ofEntries(
            entry(Soup.CONFIG.name(EIdType.EMAIL), Matching.Type.EMAIL),
            entry(Soup.CONFIG.name(EIdType.PHONE), Matching.Type.PHONE),
            entry(Soup.CONFIG.name(EIdType.LOGIN), Matching.Type.LOGIN),
            entry(Soup.CONFIG.name(EIdType.PUID), Matching.Type.PUID)
    );

    @Inject
    private YtSoupGraphService soupGraphService;

    private Pattern pattern;

    public SoupLinksIdMatcher() {
        this.pattern = createPattern();
    }

    private static Pattern createPattern() {
        String idTypesPattern = Stream.concat(
                NON_PRIVATE_ID_TYPES.stream(),
                PRIVATE_ID_TYPES_MAPPING.keySet().stream()
        ).collect(Collectors.joining("|"));

        return Pattern.compile(String.format(
                "(?<idType>%s)[=: ](?<id>\\S+) (?<targetIdType>%s)s?",
                idTypesPattern,
                idTypesPattern
        ), Pattern.CASE_INSENSITIVE);
    }

    public Optional<MatchQuery> extractMatchQuery(String query) {
        var matcher = pattern.matcher(query);
        if (matcher.matches()) {
            String idType = matcher.group("idType");
            String id = matcher.group("id");
            String targetIdType = matcher.group("targetIdType");

            MatchQuery result = new MatchQuery();
            result.id = id;
            result.idType = idType;
            result.targetIdType = targetIdType;

            return Optional.of(result);
        }
        return Optional.empty();
    }

    @Override
    public boolean matches(Service.TSearchRequest request) {
        Optional<MatchQuery> identifier = extractMatchQuery(request.getQuery());
        if (identifier.isEmpty()) {
            return false;
        }
        MatchQuery matchQuery = identifier.get();
        Identifier id = new Identifier(identifier.get().idType, matchQuery.id);
        return id.isValid();
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("yandexuid 123 logins");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
        extractMatchQuery(request.getQuery()).ifPresent(matchQuery -> {
            if (NON_PRIVATE_ID_TYPES.contains(matchQuery.targetIdType)) {
                yield.yield(Matching.NON_PRIVATE_MATCHING);
            } else if (PRIVATE_ID_TYPES_MAPPING.containsKey(matchQuery.targetIdType)) {
                yield.yield(getPrivateRole(PRIVATE_ID_TYPES_MAPPING.get(matchQuery.targetIdType)));
            } else {
                yield.yield(Roles.ADMIN);
            }
        });
    }

    private String getPrivateRole(Matching.Type idType) {
        return Matching.role(
                Matching.Privacy.PRIVATE,
                Matching.Mode.MATCHING_SIDE_BY_SIDE,
                Matching.Hashing.NON_HASHED,
                idType
        );
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        MatchQuery matchQuery = extractMatchQuery(request.getQuery()).orElseThrow();
        Identifier identifier = new Identifier(matchQuery.idType, matchQuery.id);
        String targetIdType = matchQuery.targetIdType;

        TIdType idType = Soup.CONFIG.getIdType(identifier.getType());

        Optional<Graph> maybeGraph = soupGraphService.getById(
                new GraphId(identifier.getValue(), idType.getName())
        );

        if (maybeGraph.isPresent()) {
            Graph graph = maybeGraph.get();
            List<Vertex> linkedIds = graph.getGraphComponents().get(0).getVertices();

            List<Search.TSoupLinkedIdsByType> idsByType = linkedIds.stream()
                    .filter(id -> id.getIdType().equals(targetIdType))
                    .collect(Collectors.groupingBy(Vertex::getIdType))
                    .entrySet()
                    .stream()
                    .map(idsOfType -> Search.TSoupLinkedIdsByType
                            .newBuilder()
                            .setIdType(idsOfType.getKey())
                            .addAllIds(idsOfType.getValue()
                                    .stream()
                                    .map(Vertex::getIdValue)
                                    .collect(Collectors.toList())
                            ).build()
                    ).collect(Collectors.toList());

            Search.TResponse response = createResponse()
                    .setSource("soup")
                    .setValue(createResponseValue()
                            .setSoupLinkedIds(Search.TSoupLinkedIds
                                    .newBuilder()
                                    .addAllSoupLinkedIdsByType(idsByType)
                            ).build()
                    ).build();

            yield.yield(response);
        }
    }

    private static class MatchQuery {
        private String id;
        private String idType;
        private String targetIdType;
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        Search.TMatcherDescription description = createDescription(
                Language.EN,
                "Get directly linked ids from Crypta Soup"
        );
        Search.TMatcherDescription withRoles = description.toBuilder()
                .addRoles("Крипта/Лаборатория/Матчинг/Сопоставление...")
                .build();

        yield.yield(withRoles);
    }
}
