package ru.yandex.crypta.search.keyword;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.inject.Inject;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.search.proto.KeywordDataTypeOuterClass.KeywordDataType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

public class DefaultKeywordsService implements KeywordsService {

    private final static YPath KEYWORD_TABLES_HOME = YPath.simple("//home/yabs/dict");
    private final static YPath KEYWORD_TABLE = KEYWORD_TABLES_HOME.child("Keyword");
    private final static YPath KEYWORD_DATA_TYPE_TABLE = KEYWORD_TABLES_HOME.child("KeywordDataType");
    private final YtService ytService;
    private final Cache<Integer, Map<Long, String>> keywordsCache = CacheBuilder.newBuilder()
            .expireAfterWrite(1, TimeUnit.DAYS)
            .build();

    private final Cache<Integer, Map<Long, KeywordDataType>> keywordDataTypesCache = CacheBuilder.newBuilder()
            .expireAfterWrite(1, TimeUnit.DAYS)
            .build();

    @Inject
    public DefaultKeywordsService(YtService ytService) {
        this.ytService = ytService;
    }

    private Map<Long, KeywordDataType> fetchKeywordDataTypes() {
        Map<Long, KeywordDataType> keywords = new HashMap<>();
        List<KeywordDataType> keywordsList = new ArrayList<>();
        ytService.getHahn().tables().selectRows(
                String.format("* FROM [%s]", KEYWORD_DATA_TYPE_TABLE),
                YTableEntryTypes.proto(KeywordDataType.class),
                (Consumer<KeywordDataType>) keywordsList::add
        );

        keywordsList.forEach(kw -> keywords.put(kw.getKeywordID(), kw));
        return keywords;
    }

    private Map<Long, String> fetchKeywordDescriptions() {
        Map<Long, String> keywords = new HashMap<>();
        ytService.getHahn().tables().selectRows(
                String.format("KeywordID, Description FROM [%s]", KEYWORD_TABLE),
                YTableEntryTypes.YSON,
                (each) -> {
                    keywords.put(each.getLong("KeywordID"), each.getStringO("Description").orElse(""));
                }
        );

        return keywords;
    }

    @Override
    public KeywordDataType getKeywordDataTypeById(Long keywordId) {
        Map<Long, KeywordDataType> keywordDataTypes =
                Caching.fetch(keywordDataTypesCache, 0, this::fetchKeywordDataTypes);

        return keywordDataTypes.getOrDefault(keywordId, null);
    }

    @Override
    public String getKeywordDescriptionById(Long keywordId) {
        Map<Long, String> keywords = Caching.fetch(keywordsCache, 0, this::fetchKeywordDescriptions);
        return keywords.get(keywordId);
    }

    @Override
    public Map<Long, String> getKeywordDescriptions() {
        return Caching.fetch(keywordsCache, 0, this::fetchKeywordDescriptions);
    }
}
