package ru.yandex.crypta.search.lab;

import java.util.ArrayList;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.yt.PathUtils;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class InsightsMatcher extends LabMatcher implements RegexMatcher {

    public static final String HOST_PATTERN =
            "^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\\-]*[a-zA-Z0-9])\\.)*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9\\-]*[A-Za-z0-9])$";

    @Inject
    private YtService yt;

    @Inject
    private EEnvironment environment;

    private YPath[] getPaths() {
        var rootPath = YPath.simple("//home/crypta").child(PathUtils.toPath(environment)).child("lab")
                .child("sample_user_sets");
        return new YPath[]{rootPath.child("Apps"), rootPath.child("Domains")};
    }

    private String getLogin() {
        if (environment == EEnvironment.ENV_PRODUCTION) {
            return "robot-crypta";
        } else {
            return "robot-crypta-testing";
        }
    }

    @Override
    public Pattern regex() {
        return Pattern.compile(HOST_PATTERN);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("yandex.ru");
        yield.yield("ru.yandex.taxi");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }

        for (var path : getPaths()) {
            var records = new ArrayList<JsonNode>();

            records.addAll(yt.readTableJson(path.withExact(YtReadingUtils.exact(request.getQuery())), Function.identity()));
            if (records.isEmpty()) {
                continue;
            }

            var userSetId = records.get(0).get("user_set_id").asText();

            var stats = getLabService(context.getLanguage()).getStatsFromSiberia(userSetId, getLogin());
            if (stats.isEmpty()) {
                return;
            }

            var insights = Search.TUserDataStatsResponse
                    .newBuilder()
                    .setId(request.getQuery())
                    .setStats(stats.orElseThrow().getStats())
                    .build();
            var result = createResponseValue().setUserDataStats(insights).build();
            yield.yield(createResponse().setValue(result).setSource("Siberia").build());
        }
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show domain Siberia stats"
        ));
    }

}
