package ru.yandex.crypta.search.lab;

import java.util.Optional;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.NotFoundException;
import ru.yandex.crypta.lab.LabService;
import ru.yandex.crypta.lab.proto.Sample;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class LabSampleMatcher extends LabMatcher implements RegexMatcher {
    private final static Logger LOG = LoggerFactory.getLogger(LabSampleMatcher.class);

    @Override
    public Pattern regex() {
        return Pattern.compile("(sample-.+)");
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("sample-1337");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        try {
            LabService lab = getLabService(context.getLanguage());
            var sample = lab.samples().getSample(request.getQuery());

            getMetadataResponse(sample)
                    .ifPresent(yield::yield);
            getStatsResponse(lab, sample)
                    .ifPresent(yield::yield);
        } catch (NotFoundException ex) {
            LOG.error("Not found", ex);
        }
    }

    private Optional<Search.TResponse> getMetadataResponse(Sample sample) {
        var response = createResponse();
        response.setSource("lab");
        response.getValueBuilder()
                .getLabSampleBuilder()
                .setSample(sample);

        return Optional.of(response.build());
    }

    private Optional<Search.TResponse> getStatsResponse(LabService lab, Sample sample) {
        return lab
                .getStatsFromSiberia(sample.getSiberiaUserSetId())
                .map(stats -> {
                    var response = createResponse();
                    response.setSource("lab");

                    response.getValueBuilder()
                            .getUserDataStatsBuilder()
                            .setId(sample.getId())
                            .setStats(stats.getStats());

                    return response.build();
                });
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show Crypta Lab sample Siberia stats"
        ));
    }

}
