package ru.yandex.crypta.search.links;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.common.base.Splitter;

import ru.yandex.crypta.lib.YabsFilterRecord;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Search.TParameter;

public class YabsHelper {

    private static final String newLinkMarker = "/yandex.ru/an/";

    private YabsHelper() {
    }

    public static Pattern webCountLinkRegex() {
        return Pattern.compile("https?://(yabs\\.yandex\\.ru|an\\.yandex\\.ru|yandex\\.ru/an)/(count|abuse)/.*");
    }

    public static Collection<String> webCountLinkExamples() {
        return List.of(
                "https://an.yandex.ru/count/...",
                "https://yabs.yandex.ru/count/...",
                "https://an.yandex.rui/abuse/..."
        );
    }

    public static Pattern adsdkCountLinkRegex() {
        return Pattern.compile("https?://adsdk.yandex.ru/.*");
    }

    public static Collection<String> adsdkCountLinkExamples() {
        return List.of(
                "http://adsdk.yandex.ru/..."
        );
    }

    public static Search.TCountLink.Builder decode(String encodedLink) {
        var decodedLink = YabsFilterRecord.decode(encodedLink);

        Search.TCountLink.Builder builder = Search.TCountLink.newBuilder().setValue(decodedLink);

        var parameters = parseParameters(decodedLink);
        parameters.entrySet().stream()
                .sorted(Map.Entry.comparingByKey())
                .collect(Collectors.toMap(
                        Map.Entry::getKey, Map.Entry::getValue, (oldValue, newValue) -> oldValue, LinkedHashMap::new
                ))
                .forEach((k, v) -> builder.addParameters(TParameter.newBuilder().setKey(k).setValue(v)));

        return builder;
    }

    public static Map<String, String> toMap(Search.TCountLink.Builder countLink) {
        return countLink.getParametersList()
                .stream()
                .collect(Collectors.toMap(Search.TParameter::getKey, Search.TParameter::getValue));
    }

    private static Integer getParametersPartIndex(String link) {
        return link.contains(newLinkMarker) ? 5 : 4;
    }

    private static Map<String, String> parseParameters(String decodedLink) {
        var linkParts = Splitter.on("/").splitToList(decodedLink);
        var parametersPartIndex = getParametersPartIndex(decodedLink);
        var parametersPart = linkParts.get(parametersPartIndex);

        Map<String, String> parameters = new HashMap<>();

        for (var parameter : Splitter.on(",").split(parametersPart)) {
            var keyValue = Splitter.on("=").splitToList(parameter);

            String value;
            if (keyValue.size() > 1) {
                value = keyValue.get(1);
            } else {
                value = "";
            }

            parameters.put(keyValue.get(0), value);
        }

        return parameters;
    }
}
