package ru.yandex.crypta.search.logos;

import java.io.IOException;
import java.util.Comparator;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import javax.inject.Inject;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import okhttp3.HttpUrl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import retrofit2.Retrofit;
import retrofit2.converter.jackson.JacksonConverterFactory;

import ru.yandex.crypta.clients.utils.Caching;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.lib.proto.TLogosConfig;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class LogosMatcher implements RegexMatcher {

    private static final Logger LOG = LoggerFactory.getLogger(LogosMatcher.class);

    private Cache<String, LogosGraph> cache =
            CacheBuilder.newBuilder()
                    .expireAfterWrite(5, TimeUnit.MINUTES)
                    .build();

    @Inject
    private TLogosConfig config;

    @Override
    public Pattern regex() {
        return Pattern.compile("(logos )?(?<name>[a-z0-9]+)", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("JoinedEFHLog");
        yield.yield("logos efh");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }
        var name = matcher.group("name");
        var api = getLogosApi();
        var logosGraphId = fetchLatestGraphMeta(api).orElseThrow();
        var logosGraph = Caching.fetch(cache, logosGraphId, () -> fetchGraph(api, logosGraphId).orElseThrow());

        logosGraph.getNodes().forEach(node -> {
            if (node.getTitle().toLowerCase().contains(name.toLowerCase())) {
                var response = createResponse();
                var logosUrl = String.format("https://logos.yandex-team.ru/#/graph/%s/node/%s", logosGraphId, node.getId());
                var logosDocUrl = String.format("https://logos.yandex-team.ru/docs/md_schema/%s.html", node.getTitle());
                response.getValueBuilder()
                        .getLogosNodeBuilder()
                        .setTitle(node.getTitle())
                        .setLogosUrl(logosUrl)
                        .setLogosDocUrl(logosDocUrl)
                        .setYtUrl(node.getUrl());

                response.setSource("Logos");
                yield.yield(response.build());
            }
        });
    }

    private Optional<LogosGraph> fetchGraph(LogosApi api, String id) {
        try {
            var response = api.getGraph(id).execute();
            return Optional.ofNullable(response.body());
        } catch (IOException e) {
            LOG.warn("Failed to fetch graph {}", id, e);
            return Optional.empty();
        }
    }

    private Optional<String> fetchLatestGraphMeta(LogosApi api) {
        try {
            var response = api.getGraphs().execute();
            if (!response.isSuccessful()) {
                return Optional.empty();
            }
            return response
                    .body()
                    .stream()
                    .map(LogosGraphMeta::getId)
                    .max(Comparator.naturalOrder());
        } catch (IOException e) {
            LOG.warn("Failed to fetch graph meta", e);
            return Optional.empty();
        }
    }

    private LogosApi getLogosApi() {
        var url = new HttpUrl.Builder()
                .scheme("https")
                .host(config.getHost())
                .build();
        return new Retrofit.Builder()
                .baseUrl(url)
                .addConverterFactory(JacksonConverterFactory.create())
                .build()
                .create(LogosApi.class);
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Get Logos log meta (YT path, docs)"
        ));
    }

}
