package ru.yandex.crypta.search.lookalike;

import java.util.Comparator;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.inject.Inject;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lab.siberia.SiberiaClient;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.lookalike.proto.TYtNodeNames;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.crypta.search.siberia.YtToSiberiaStatsMatcher;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.misc.lang.number.UnsignedLong;

public class LalMatcher extends YtToSiberiaStatsMatcher implements RegexMatcher {

    private static final YPath VERSIONS_DIR = YPath.simple("//home/crypta/production/lookalike/scopes/direct/versions");

    @Inject
    private YtService yt;

    @Inject
    public LalMatcher(
            SiberiaClient siberiaClient
    )
    {
        super(siberiaClient);
    }

    @Override
    public Pattern regex() {
        return Pattern.compile("^(lal|лал) (?<id>[0-9]+)$", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("lal 22");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }

        var id = UnsignedLong.valueOf(Long.parseUnsignedLong(matcher.group("id")));

        ListF<YTreeNode> key = Cf.arrayList(YTree.stringNode("lookalike"), YTree.unsignedLongNode(id.longValue()));
        var path = findLatestLalExportPath().withExact(YtReadingUtils.exact(key));

        yield.yield(describeFromYt(path, "yandexuid", "yandexuid"));
    }

    private YPath findLatestLalExportPath() {
        var nodes = yt.getHahn().cypress().list(VERSIONS_DIR).stream()
                .sorted(Comparator.comparing(YTreeStringNode::getValue).reversed())
                .collect(Collectors.toList());

        for (var node : nodes) {
            var path = VERSIONS_DIR.child(node.getValue()).child(TYtNodeNames.newBuilder().build().getLalExportTable());
            if (yt.getHahn().cypress().exists(path)) {
                return path;
            }
        }

        throw Exceptions.internal("No lal export found");
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show LaL-segment Siberia stats"
        ));
    }
}
