package ru.yandex.crypta.search.lookalike;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.regex.Pattern;

import javax.inject.Inject;

import com.google.protobuf.InvalidProtocolBufferException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.crypta.audience.proto.TUserDataStats;
import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.lookalike.proto.TLalState;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.lab.LabMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

public class LalParentMatcher extends LabMatcher implements RegexMatcher {

    private static final YPath LALS_TABLE = YPath.simple("//home/crypta/production/lookalike/scopes/direct/lals");

    @Inject
    private YtService yt;

    @Override
    public Pattern regex() {
        return Pattern.compile("^lal parent (?<id>[0-9]+)$", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("lal 22");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }
        var id = matcher.group("id");

        var lalStateRaw = getLalStateRaw(id);
        var userDataStats = getUserDataStats(lalStateRaw);
        var stats = getLabService(context.getLanguage()).getSimpleSampleStats(userDataStats);

        var response = createResponse();
        response.setSource("siberia");
        response.getValueBuilder()
                .getUserDataStatsBuilder()
                .setId(id)
                .setStats(stats);
        yield.yield(response.build());
    }

    private TUserDataStats getUserDataStats(byte[] value) {
        try {
            return TLalState.parseFrom(value).getDescription().getUserDataStats();
        } catch (InvalidProtocolBufferException e) {
            throw Exceptions.internal(e.getMessage());
        }
    }

    private byte[] getLalStateRaw(String id) {
        List<YTreeMapNode> results = new ArrayList<>();

        YTreeMapNode key = YTree.mapBuilder()
                .key("key")
                .value(id)
                .buildMap();

        yt.getHahn().tables().lookupRows(
                LALS_TABLE,
                YTableEntryTypes.YSON,
                Cf.list(key),
                YTableEntryTypes.YSON,
                (Consumer<YTreeMapNode>) results::add
        );

        if (results.size() > 0) {
            return results.get(0).getBytes("value");
        } else {
            throw Exceptions.notFound();
        }
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Have no idea what is 'parent'"
        ));
    }
}
