package ru.yandex.crypta.search.misc;

import java.util.ArrayList;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.ws.rs.container.ContainerRequestContext;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.lang.number.UnsignedLong;

public class AdhocYtTableMatcher implements RegexMatcher {

    public static final String YANDEXUID_PLACEHOLDER = "YANDEXUID";
    public static final String YANDEXUID_COOKIE = "yandexuid";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    @Inject
    private YtService yt;

    @Inject
    private Provider<ContainerRequestContext> requestContext;

    @Override
    public Pattern regex() {
        return Pattern.compile("(?<path>//home/crypta/public/adhoc/.*)\\[(?<key>.*)\\]");
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("//home/crypta/public/adhoc/ExperimentTable[123]");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (matcher.matches()) {
            var path = YPath.simple(matcher.group("path"));
            var key = matcher.group("key");
            if (YANDEXUID_PLACEHOLDER.equals(key)) {
                var yandexuid = requestContext.get().getCookies().get(YANDEXUID_COOKIE);
                if (Objects.isNull(yandexuid)) {
                    return;
                }
                key = yandexuid.getValue();
            }

            var records = new ArrayList<JsonNode>();

            records.addAll(yt.readTableJson(path.withExact(YtReadingUtils.exact(key)), Function.identity()));
            tryParse(key).ifPresent(ulKey -> {
                records.addAll(yt.readTableJson(path.withExact(YtReadingUtils.exact(ulKey)), Function.identity()));
            });

            records.stream()
                    .map(each -> createResponse()
                            .setSource("YT")
                            .setValue(createResponseValue().setJson(backIntoJson(each)))
                            .build())
                    .forEach(yield::yield);
        }
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(Language.EN, "Read YT table by key"));
        yield.yield(createDescription(Language.RU, "Прочитать таблицу YT по ключу"));
    }

    private String backIntoJson(JsonNode jsonNode) {
        try {
            return OBJECT_MAPPER.writeValueAsString(jsonNode);
        } catch (JsonProcessingException e) {
            return "<invalid_json>";
        }
    }

    private Optional<UnsignedLong> tryParse(String key) {
        try {
            return Optional.of(UnsignedLong.valueOf(Long.parseUnsignedLong(key)));
        } catch (NumberFormatException e) {
            return Optional.empty();
        }
    }

}
