package ru.yandex.crypta.search.misc;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.inject.Inject;

import NExperiments.Experiment;
import com.google.common.base.Splitter;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.experiments.Experiments;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.proto.TYtConfig;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class ExpStatsMatcher implements RegexMatcher {

    private static final Logger LOG = LoggerFactory.getLogger(ExpStatsMatcher.class);
    private static final Pattern KEYS_REGEX = Pattern.compile("--key (?<keys>[a-z,]+)", Pattern.CASE_INSENSITIVE);

    private final TYtConfig ytConfig;
    private final OkHttpClient httpClient;
    private final boolean useHttpExpStatsService = true;
    private final HttpUrl expStatsServiceUrl;

    @Inject
    public ExpStatsMatcher(TYtConfig ytConfig, EEnvironment environment) {
        this.ytConfig = ytConfig;
        this.httpClient = new OkHttpClient.Builder()
                .readTimeout(5, TimeUnit.MINUTES).build();

        if (EEnvironment.ENV_PRODUCTION.equals(environment)) {
            expStatsServiceUrl = HttpUrl.get("http://exp-stats.crypta.yandex.net/exp_stats");
        } else {
            expStatsServiceUrl = HttpUrl.get("http://exp-stats-test.crypta.yandex.net/exp_stats");
        }
    }

    @Override
    public Pattern regex() {
        return Pattern.compile("(\\./)?(?<query>exp_stats .*)", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("./exp_stats --exp 6644 --range last_1_days --from mq");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        Experiment.TExperimentStats.Builder stats;
        if (useHttpExpStatsService) {
            HttpUrl url = expStatsServiceUrl.newBuilder()
                    .addQueryParameter("query", request.getQuery())
                    .addQueryParameter("serialize", "string")
                    .build();

            LOG.info("Calling {} to get exp_stats result", url);

            Request call = new Request.Builder().url(url).build();
            try (Response response = httpClient.newCall(call).execute()) {
                if (response.isSuccessful()) {
                    var protoBytes = Objects.requireNonNull(response.body()).bytes();

                    stats = Experiment.TExperimentStats
                            .newBuilder()
                            .mergeFrom(protoBytes);
                } else {
                    throw Exceptions.illegal(response.code() + ": " + response.message());
                }

            } catch (IOException e) {
                throw Exceptions.unchecked(e);
            }
        } else {
            LOG.info("Calling exp_stats natively");
            stats = Experiments.getExperimentStats(
                    request.getQuery(), ytConfig.getToken()
            ).toBuilder();
        }

        // TODO: exp-stats must fill keys by itself
        List<String> keys = extractKeys(request);
        stats.addAllKeys(keys);

        var response = createResponse().setSource("ExpStats");

        response.getMetaBuilder().setDetailsUrl(
                "https://a.yandex-team.ru/arc/trunk/arcadia/ads/bsyeti/exp_stats"
        );

        response.getValueBuilder()
                .getExpStatsBuilder()
                .setStats(stats);
        yield.yield(response.build());
    }

    @NotNull
    private List<String> extractKeys(Service.TSearchRequest request) {
        List<String> keys = new ArrayList<>();
        // these keys are automatically added even though it's not in the query
        keys.add("exp");
        keys.add("exp_description");
        keys.add("rsya");

        Matcher matcher = KEYS_REGEX.matcher(request.getQuery());
        if (matcher.find()) {
            for (String foundKey : Splitter.on(",").split(matcher.group("keys"))) {
                if (!keys.contains(foundKey)) {
                    keys.add(foundKey);
                }
            }
        }
        return keys;
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Run exp_stats query in the cloud!"
        ));
    }

    @Override
    public Duration getCacheTtl() {
        return Duration.ofHours(1);
    }
}
