package ru.yandex.crypta.search.misc;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.regex.Pattern;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class HashFunctionMatcher implements RegexMatcher {

    private static final String md5functionName = "MD5";
    private static final String sha256functionName = "SHA-256";

    private static final Map<String, String> functionNamesMap = Map.of(
            "MD5", md5functionName,
            "SHA256", sha256functionName
    );

    @Override
    public Pattern regex() {
        return Pattern.compile("^(md[\\-]?5|sha[\\-]?256)\\s.*$", Pattern.CASE_INSENSITIVE);
    }

    private String getDigest(String functionName, String functionArgument) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(functionName);
        } catch (NoSuchAlgorithmException e) {
            throw Exceptions.unsupported();
        }

        byte[] hash = digest.digest(functionArgument.getBytes(StandardCharsets.UTF_8));

        StringBuilder result = new StringBuilder();
        for (byte b : hash) {
            result.append(Integer.toString((b & 0xff) + 0x100, 16).substring(1));
        }

        return result.toString();
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("md5 email@yandex.ru");
        yield.yield("sha256 +71234567890");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        String[] parts = request.getQuery().split(" ", 2);

        String normalizedFunctionName = normalizeFunctionName(parts[0]);
        String functionArgument = "";
        if (parts.length > 1) {
            functionArgument = parts[1];
        }

        String digest = getDigest(functionNamesMap.get(normalizedFunctionName), functionArgument);

        yield.yield(createResponse()
                .setValue(createResponseValue().setText(digest))
                .setSource("api")
                .build());
    }

    private String normalizeFunctionName(String functionName) {
        return functionName.replaceAll("\\-", "").toUpperCase();
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Apply hash function"
        ));
    }

}
