package ru.yandex.crypta.search.misc;

import java.util.function.Function;
import java.util.regex.Pattern;

import javax.inject.Inject;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.Nullable;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.lib.yt.YtReadingUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.search.Matcher;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;

public class HistoryMatcher implements RegexMatcher {

    private static final YPath HISTORY_TABLES = YPath.simple("//home/crypta/production/history/daily");
    private final ObjectMapper mapper = new ObjectMapper();

    @Inject
    private YtService yt;

    @Override
    public Pattern regex() {
        return Pattern.compile("history (?<id>\\d*)$");
    }

    @Override
    public void examples(Matcher.Yield<String> yield) {
        yield.yield("history <CryptaID>");
    }

    @Override
    public void roles(Service.TSearchRequest request, Matcher.Yield<String> yield) {
        yield.yield(Roles.Portal.PROFILE);
    }

    @Override
    public void process(Service.TSearchRequest request, Matcher.Context context, Matcher.Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }

        var cryptaId = Long.parseUnsignedLong(matcher.group("id"));
        var tables = yt.getHahn().cypress().list(HISTORY_TABLES);
        for (var eachTable : tables) {
            var fqPath = HISTORY_TABLES.child(eachTable.getValue());
            var data = yt.readTableJsonUtf8(getPathWithId(fqPath, cryptaId), Function.identity());
            String jsonResponse = castToJson(data);
            yield.yield(createResponse().setValue(createResponseValue().setUserHistory(jsonResponse)).build());
        }
    }

    @Nullable
    private String castToJson(ru.yandex.bolts.collection.ListF<com.fasterxml.jackson.databind.JsonNode> data) {
        try {
            return mapper.writeValueAsString(data);
        } catch (JsonProcessingException e) {
            return "[]";
        }
    }

    @Override
    public void description(Matcher.Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show history of CryptaID"
        ));
    }

    private YPath getPathWithId(YPath path, long cryptaId) {
        return path.withExact(YtReadingUtils.exact(YTree.unsignedLongNode(cryptaId)));
    }
}
