package ru.yandex.crypta.search.misc;

import java.time.Duration;
import java.util.regex.Pattern;

import javax.inject.Inject;
import javax.inject.Provider;
import javax.ws.rs.core.SecurityContext;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.graph.api.model.ids.GraphId;
import ru.yandex.crypta.idm.Roles;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.crypta.service.ltp.client.LtpViewerClient;

public class LtpHistoryMatcher implements RegexMatcher {

    private final LtpViewerClient ltpViewerClient;
    private final Provider<SecurityContext> securityContextProvider;

    @Inject
    public LtpHistoryMatcher(LtpViewerClient ltpViewerClient, Provider<SecurityContext> securityContextProvider) {
        this.ltpViewerClient = ltpViewerClient;
        this.securityContextProvider = securityContextProvider;
    }

    @Override
    public Pattern regex() {
        return Pattern.compile(
                "(history_v2|ltp_v2|history|ltp) " +
                        "(:?(?<IdType>[a-zA-Z_]*) )?" +  // optional type
                        "(?<Id>[\\w-]+)" +
                        "(:? (?<FromDate>\\d{4}-\\d{2}-\\d{2}))?" +   // optional date
                        "(:? (?<ToDate>\\d{4}-\\d{2}-\\d{2}))?");     // optional date
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("history_v2 <CryptaID>");
        yield.yield("history_v2 puid 123");
        yield.yield("history_v2 puid 123 <FromDate> <ToDate>");
        yield.yield("history_v2 puid 123 2020-11-11 2020-11-15");
        yield.yield("history_v2 puid 123 2020-11-11");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
        yield.yield(Roles.Portal.LONG_TERM_PROFILE);
    }

    private boolean isUnsignedLong(String id) {
        try {
            long ignored = Long.parseUnsignedLong(id);
            return true;
        } catch (NumberFormatException e) {
            return false;
        }
    }

    @Override
    public void process(Service.TSearchRequest request, Context context,
                        Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }

        String id = matcher.group("Id");
        String idType = matcher.group("IdType");

        if (idType == null) {
            if (isUnsignedLong(id)) {
                idType = GraphId.CRYPTA_ID_TYPE;
            } else {
                return;
            }
        }

        String login = securityContextProvider.get().getUserPrincipal().getName();

        String fromDate = matcher.group("FromDate");
        String toDate = matcher.group("ToDate");

        ltpViewerClient.preloadHistory(idType, id, login, fromDate, toDate);
        var ltpResponse = createResponseValue();
        ltpResponse.getLtpProfileResponseBuilder()
                .setResultID(String.format("%s:%s", idType, id))
                .setVersion(2)
                .setId(id)
                .setIdType(idType)
                .setFromDate(fromDate == null ? "" : fromDate)
                .setToDate(toDate == null ? "" : toDate);
        yield.yield(createResponse().setValue(ltpResponse).build());
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show LTP history of user by ID"
        ));
    }

    @Override
    public Duration getCacheTtl() {
        return Duration.ZERO;
    }
}
