package ru.yandex.crypta.search.misc;

import java.util.regex.Pattern;

import javax.inject.Inject;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

public class SelectTypeMatcher implements RegexMatcher {

    private static final YPath SELECT_TYPE_TABLE = YPath.simple("//home/yabs/dict/SelectType");

    @Inject
    private YtService yt;

    @Override
    public Pattern regex() {
        return Pattern.compile("^(st|select_type|select type|ст)( |:|=)(?<id>[0-9]+)$", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("ST 82");
        yield.yield("select type 82");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());
        if (!matcher.matches()) {
            return;
        }
        var id = matcher.group("id");

        yt.getHahn().tables().selectRows(getQuery(id), YTableEntryTypes.YSON, each -> {
            var response = createResponse().setSource("YT");
            var builder = response.getValueBuilder().getSelectTypeBuilder();

            builder.setId(each.getLong("SelectType"));
            builder.setDescription(each.getString("Description"));
            each.asMap().forEach((key, value) -> {
                if (value.isEntityNode()) {
                    builder.putParameters(key, "<null>");
                }
                if (value.isDoubleNode()) {
                    builder.putParameters(key, Double.toString(value.doubleValue()));
                }
                if (value.isIntegerNode()) {
                    builder.putParameters(key, Long.toUnsignedString(value.longValue()));
                }
                if (value.isStringNode()) {
                    builder.putParameters(key, value.stringValue());
                }
            });
            yield.yield(response.build());
        });
    }

    private String getQuery(String id) {
        return String.format("* FROM [%s] WHERE SelectType = %s", SELECT_TYPE_TABLE.toString(), id);
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show select type description"
        ));
    }

}
