package ru.yandex.crypta.search.misc;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Locale;
import java.util.regex.Pattern;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class UnixTimestampMatcher implements RegexMatcher {

    private static final String DATE_FORMAT_PATTERN = "dd MMM yyyy HH:mm:ss Z";

    @Override
    public Pattern regex() {
        return Pattern.compile("^[0-9]{1,10}$");
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield(String.valueOf(Instant.now().getEpochSecond()));
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var milliseconds = Long.valueOf(request.getQuery()) * 1000;

        var humanTimestamp = Instant.ofEpochMilli(milliseconds);

        DateTimeFormatter formatter =
                DateTimeFormatter
                        .ofPattern(DATE_FORMAT_PATTERN)
                        .withLocale(Locale.getDefault())
                        .withZone(ZoneId.systemDefault());

        String formattedTime = formatter.format(humanTimestamp);

        yield.yield(createResponse()
                .setValue(createResponseValue().setText(formattedTime))
                .setSource("api")
                .build());
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Decode unix-time"
        ));
    }

}
