package ru.yandex.crypta.search.prism;

import java.util.Optional;
import java.util.regex.Pattern;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lab.proto.TSimpleSampleStats;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.lab.LabMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class PrismMatcher extends LabMatcher implements RegexMatcher {
    private static final YPath STATS_TABLE =
            YPath.simple("//home/crypta/production/profiles/monitorings/prism/clusters_daily_stats");

    @Override
    public Pattern regex() {
        return Pattern.compile(
            "^(prism|призма) (?<cluster>p[1-5]):(?<date>\\d{4}-\\d{2}-\\d{2})$", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("prism p1:2020-04-28");
        yield.yield("призма p5:2020-04-28");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());

        if (!matcher.matches()) {
            return;
        }

        var date = matcher.group("date");
        var cluster = matcher.group("cluster");
        var id = date + '-' + cluster;

        TSimpleSampleStats stats;

        var result = getLabService(context.getLanguage()).getStatsFromYt(STATS_TABLE, id, Optional.empty());
        if (result.isEmpty()) {
            throw Exceptions.notFound();
        }
        stats = result.get();

        yield.yield(getResponse(id, stats));
    }

    private Search.TResponse getResponse(String id, TSimpleSampleStats stats) {
        var response = createResponse();
        response.setSource("Prism");
        response.getValueBuilder()
                .getUserDataStatsBuilder()
                .setId(id)
                .setStats(stats);
        return response.build();
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show Stats for Prism cluster"
        ));
    }
}
