package ru.yandex.crypta.search.siberia;

import java.util.regex.Pattern;

import ru.yandex.crypta.common.Language;
import ru.yandex.crypta.lab.proto.TSimpleSampleStats;
import ru.yandex.crypta.search.RegexMatcher;
import ru.yandex.crypta.search.lab.LabMatcher;
import ru.yandex.crypta.search.proto.Search;
import ru.yandex.crypta.search.proto.Service;

public class SiberiaMatcher extends LabMatcher implements RegexMatcher {
    @Override
    public Pattern regex() {
        return Pattern.compile(
                "^(user_set|userset|юзерсет) (?<id>[0-9]+)(?<args>.*)$", Pattern.CASE_INSENSITIVE);
    }

    @Override
    public void examples(Yield<String> yield) {
        yield.yield("user_set 123456789");
        yield.yield("user_set 123456789 version=v1_1");
        yield.yield("user_set 123456789 yt_path=//home/crypta/stats");
        yield.yield("user_set 123456789 version=v1_1 yt_path=//home/crypta/stats");
    }

    @Override
    public void roles(Service.TSearchRequest request, Yield<String> yield) {
    }

    @Override
    public void process(Service.TSearchRequest request, Context context, Yield<Search.TResponse> yield) {
        var matcher = regex().matcher(request.getQuery());

        if (!matcher.matches()) {
            return;
        }

        var id = matcher.group("id");
        var args = matcher.group("args");
        var params = ParamsParser.parseArgs(args);
        var customUserDataStatsVersion = params.getVersion();
        var ytPath = params.getYtPath();

        TSimpleSampleStats stats;

        if (ytPath.isPresent()) {
            var result = getLabService(context.getLanguage()).getStatsFromYt(ytPath.get(), id, customUserDataStatsVersion);
            if (result.isEmpty()) {
                return;
            }
            stats = result.get();
        } else {
            var result = getLabService(context.getLanguage()).getStatsFromSiberia(id, customUserDataStatsVersion);
            if (result.isEmpty()) {
                return;
            }
            stats = result.get().getStats();
        }

        yield.yield(getResponse(id, stats));
    }

    private Search.TResponse getResponse(String id, TSimpleSampleStats stats) {
        var response = createResponse();
        response.setSource("Siberia");
        response.getValueBuilder()
                .getUserDataStatsBuilder()
                .setId(id)
                .setStats(stats);
        return response.build();
    }

    @Override
    public void description(Yield<Search.TMatcherDescription> yield) {
        yield.yield(createDescription(
                Language.EN, "Show Siberia stats for user_set"
        ));
    }
}
