package ru.yandex.crypta.search.util;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import com.google.common.base.CaseFormat;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ProtocolMessageEnum;

import ru.yandex.bolts.function.Function;
import ru.yandex.crypta.common.exception.Exceptions;

public class ParamsParsing {
    public interface Consumer {
        <T> void consume(String fieldName, T value);
        <T> void consumeMultiple(String fieldName, Stream<T> values);
    }

    public interface FieldParser {
        void parse(Map<String, List<String>> args, Consumer consumer);
        String usage();
    }

    public abstract static class AttributeParser implements FieldParser {
        protected final String fieldName;
        protected final String mapKey;

        protected AttributeParser(String fieldName) {
            this.fieldName = fieldName;
            this.mapKey = fieldNameToMapKey(fieldName);
        };

        @Override
        public void parse(Map<String, List<String>> args, Consumer consumer) {
            var rawValues = args.remove(this.mapKey);
            if (rawValues == null) {
                return;
            }

            if (rawValues.size() > 1) {
                throw Exceptions.wrongRequestException(MessageFormat.format("{0} is single value, but multiple values were provided: {1}", fieldName, rawValues), "BAD_REQUEST");
            }

            var rawValue = rawValues.get(0);
            var value = getValue(rawValue);

            consumer.consume(fieldName, value);
        }

        public abstract Object getValue(String rawValue);
    }

    public static class MappingAttributeParser extends AttributeParser {
        private final Map<String, Object> mapping;

        public MappingAttributeParser(String fieldName, Map<String, Object> mapping) {
            super(fieldName);
            this.mapping = mapping;
        };

        @Override
        public String usage() {
            return MessageFormat.format("{0}={1}", mapKey, getValueHint());
        }

        @Override
        public Object getValue(String rawValue) {
            var value = mapping.get(rawValue);
            if (value == null) {
                throw Exceptions.wrongRequestException(MessageFormat.format("{0} has wrong value \"{1}\", valid values: {2}", mapKey, rawValue, mapping.keySet()), "BAD_REQUEST");
            }
            return value;
        }

        public String getValueHint() {
            return String.join("|", mapping.keySet());
        }

    }

    public static class EnumAttributeParser extends MappingAttributeParser {
        private static final String UNKNOWN = "unknown";

        public static Function<String, String> getDefaultConverter() {
            return (x) -> CaseFormat.UPPER_UNDERSCORE.to(CaseFormat.LOWER_HYPHEN, x);
        };

        public <T extends ProtocolMessageEnum> EnumAttributeParser(String fieldName, T[] values) {
            this(fieldName, values, getDefaultConverter());
        };

        public <T extends ProtocolMessageEnum> EnumAttributeParser(String fieldName, T[] values, Function<String, String> transformer) {
            super(fieldName, buildMap(values, transformer));
        }

        private static <T extends ProtocolMessageEnum> Map<String, Object> buildMap(T[] values, Function<String, String> transformer) {
            var builder = ImmutableMap.<String, Object>builder();
            for (var i = 0; i < values.length-1; i++) {
                var value = values[i];

                if (value.getNumber() == 0) {
                    builder.put(UNKNOWN, value.getValueDescriptor());
                } else {
                    builder.put(transformer.apply(value.toString()), value.getValueDescriptor());
                }
            }
            return builder.build();
        };
    }

    public static class IntAttributeParser extends AttributeParser {
        public IntAttributeParser(String fieldName) {
            super(fieldName);
        }

        @Override
        public String usage() {
            return MessageFormat.format("{0}=<int>", mapKey);
        }

        @Override
        public Object getValue(String rawValue) {
            try {
                return Integer.parseInt(rawValue);
            } catch (NumberFormatException e) {
                throw Exceptions.wrongRequestException(MessageFormat.format("{0} should be integer, got ''{1}''", mapKey, rawValue), "BAD_REQUEST");
            }
        }
    }

    public static class StringAttributeParser extends AttributeParser {
        public StringAttributeParser(String fieldName) {
            super(fieldName);
        }

        @Override
        public String usage() {
            return MessageFormat.format("{0}=<value>", mapKey);
        }

        @Override
        public Object getValue(String rawValue) {
            return rawValue;
        }
    }

    public static class RegexAttributeParser extends AttributeParser {
        private String regex;

        public RegexAttributeParser(String fieldName, String regex) {
            super(fieldName);
            this.regex = regex;
        }

        @Override
        public String usage() {
            return MessageFormat.format("{0}=<regexp>", mapKey);
        }

        @Override
        public Object getValue(String rawValue) {
            if (rawValue.matches(regex)) {
                return rawValue;
            } else {
                throw Exceptions.wrongRequestException(MessageFormat.format("{0} should match ''{1}'', got ''{2}''", mapKey, regex, rawValue), "BAD_REQUEST");
            }
        }
    }

    public static class RepeatedFieldParser implements FieldParser {
        private final String fieldName;
        private final String mapKey;
        private final String valueHint;
        private final Function<String, Object> parser;

        public RepeatedFieldParser(String fieldName) {
            this(fieldName, (x) -> x, "<string>");
        }

        public RepeatedFieldParser(String fieldName, Function<String, Object> parser, String valueHint) {
            this.fieldName = fieldName;
            this.parser = parser;
            this.valueHint = valueHint;

            var nameInMap = fieldNameToMapKey(fieldName);

            if (nameInMap.endsWith("ies")) {
                nameInMap = nameInMap.substring(0, nameInMap.length()-3) + "y";
            } else if (nameInMap.endsWith("s")) {
                nameInMap = nameInMap.substring(0, nameInMap.length()-1);
            }

            this.mapKey = nameInMap;
        };

        @Override
        public void parse(Map<String, List<String>> args, Consumer consumer) {
            var rawValues = args.remove(mapKey);
            if (rawValues == null) {
                return;
            }

            consumer.consumeMultiple(fieldName, rawValues.stream().map(parser));
        }

        @Override
        public String usage() {
            return MessageFormat.format("{0}={1} [{0}={1}}]...", mapKey, valueHint);
        }
    }

    public static class EnumRepeatedFieldParser implements FieldParser {
        private final RepeatedFieldParser repeatedFieldParser;
        private final EnumAttributeParser enumParser;

        public <T extends ProtocolMessageEnum> EnumRepeatedFieldParser(String fieldName, T[] values) {
            this(fieldName, values, EnumAttributeParser.getDefaultConverter());
        };

        public <T extends ProtocolMessageEnum> EnumRepeatedFieldParser(String fieldName, T[] values, Function<String, String> transformer) {
            enumParser = new EnumAttributeParser(fieldName, values, transformer);
            repeatedFieldParser = new RepeatedFieldParser(fieldName, (x) -> enumParser.getValue(x), enumParser.getValueHint());
        }

        @Override
        public String usage() {
            return repeatedFieldParser.usage();
        }

        @Override
        public void parse(Map<String, List<String>> args, Consumer consumer) {
            repeatedFieldParser.parse(args, consumer);
        }
    }

    public static Map<String, List<String>> parseArgs(String rawArgs) {
        var argsList = Splitter.on(Pattern.compile("\\s+")).splitToList(rawArgs.trim());
        var args = new HashMap<String, List<String>>();
        for (var rawPair: argsList) {
            if (rawPair.isEmpty()) {
                continue;
            }

            var kv = rawPair.split("=", 2);
            if (kv.length != 2) {
                throw Exceptions.wrongRequestException("arguments should be provided in key=value format", "BAD_REQUEST");
            }

            var key = kv[0].trim();
            var value = kv[1].trim();

            args.putIfAbsent(key, new LinkedList<String>());
            args.get(key).add(value.toLowerCase());
        }
        return args;
    }

    public static void parseArgs(String rawArgs, List<FieldParser> parsers, Consumer consumer) {
        var args = parseArgs(rawArgs);

        for (var parser: parsers) {
            parser.parse(args, consumer);
        }

        if (!args.isEmpty()) {
            throw Exceptions.wrongRequestException(MessageFormat.format("Unrecognized keys: {0}", args.keySet()), "BAD_REQUEST");
        }
    }

    public static String fieldNameToMapKey(String fieldName) {
        return fieldName.replaceAll("([a-z])([A-Z]+)", "$1_$2").toLowerCase();
    }
}
