import concurrent.futures
import logging
import time

import retry


logger = logging.getLogger(__name__)

USER_SET_ID = "user_set_id"


def get_stats(yt_client, siberia_client, tvm_ticket, path, threads=50, retry_timeout=5):
    @retry.retry(tries=5, delay=1)
    def get_stats(user_set_id):
        logger.info("Getting stats for user_set_id=%s", user_set_id)
        response = siberia_client.user_sets_get_stats(user_set_id, tvm_ticket)
        logger.info("Got API response for user_set_id=%s", user_set_id)
        return user_set_id, response

    user_sets = [row[USER_SET_ID] for row in yt_client.read_table(path)]
    result = {}

    while user_sets:
        logger.info("Checking %s user sets", len(user_sets))

        with concurrent.futures.ThreadPoolExecutor(threads) as executor:
            futures = [executor.submit(get_stats, user_set_id) for user_set_id in user_sets]
            concurrent.futures.wait(futures)

        next_user_sets = []

        for future in futures:
            user_set_id, stats = future.result()
            if stats.Info.Ready:
                result[user_set_id] = stats
            else:
                next_user_sets.append(user_set_id)

        user_sets = next_user_sets

        logger.info("%s user sets left", len(user_sets))

        if user_sets:
            time.sleep(retry_timeout)

    return result
