#include "rule_applier.h"

#include <util/generic/algorithm.h>

using namespace NCrypta::NSiberia;

TRuleApplier::TRuleApplier(const TExpressionPtr& expression)
    : Expression(expression)
{
    Y_ENSURE(Expression != nullptr);
}

bool TRuleApplier::IsSatisfy(const NYT::TNode::TMapType& node) const {
    return ApplyRule(Expression, node);
}

bool NCrypta::NSiberia::ApplyRule(const TExpressionPtr& expression, const NYT::TNode::TMapType& node) {
    TApplyVisitor visitor(node);
    expression->accept(visitor);
    return visitor.Result;
}

TApplyVisitor::TApplyVisitor(const NYT::TNode::TMapType& node)
    : Node(node)
    , Result(false)
{}

void TApplyVisitor::visitSimpleExpression(const TSimpleExpression& expression) {
    const auto& it = Node.find(expression.Field);

    if (it == Node.end()) {
        Result = false;
        return;
    }

    const auto& value = it->second;

    switch (expression.Op) {
        case TSimpleExpression::EOp::Equal:
            Result = (value == expression.Value);
            break;
        case TSimpleExpression::EOp::NotEqual:
            Result = (value != expression.Value);
            break;
        case TSimpleExpression::EOp::Less:
            Result = NYT::NNodeCmp::operator<(value, expression.Value);
            break;
        case TSimpleExpression::EOp::LessOrEqual:
            Result = NYT::NNodeCmp::operator<=(value, expression.Value);
            break;
        case TSimpleExpression::EOp::More:
            Result = NYT::NNodeCmp::operator>(value, expression.Value);
            break;
        case TSimpleExpression::EOp::MoreOrEqual:
            Result = NYT::NNodeCmp::operator>=(value, expression.Value);
            break;
        case TSimpleExpression::EOp::Contains:
            if (!value.IsList()) {
                Result = false;
            } else {
                const auto& list = value.AsList();
                Result = (Find(list.begin(), list.end(), expression.Value) != list.end());
            }
            break;
        case TSimpleExpression::EOp::StartsWith:
            if (!value.IsString()) {
                Result = false;
            } else {
                Result = value.AsString().StartsWith(expression.Value.AsString());
            }
            break;
        case TSimpleExpression::EOp::EndsWith:
            if (!value.IsString()) {
                Result = false;
            } else {
                Result = value.AsString().EndsWith(expression.Value.AsString());
            }
            break;
    }
}

void TApplyVisitor::visitUnaryExpression(const TUnaryExpression& expression) {
    bool childResult = ApplyRule(expression.Child, Node);

    switch (expression.Op) {
        case TUnaryExpression::EOp::Not:
            Result = !childResult;
            break;
    }
}

void TApplyVisitor::visitBinaryExpression(const TBinaryExpression& expression) {
    bool leftResult = ApplyRule(expression.Left, Node);

    if (leftResult && expression.Op == TBinaryExpression::EOp::Or) {
        Result = true;
        return;
    } else if (!leftResult && expression.Op == TBinaryExpression::EOp::And) {
        Result = false;
        return;
    }

    bool rightResult = ApplyRule(expression.Right, Node);

    switch (expression.Op) {
        case TBinaryExpression::EOp::Or:
            Result = (leftResult || rightResult);
            break;
        case TBinaryExpression::EOp::And:
            Result = (leftResult && rightResult);
            break;
    }
}
