#pragma once

#include <library/cpp/yson/node/node.h>

#include <util/generic/ptr.h>
#include <util/generic/string.h>

namespace NCrypta::NSiberia {
    class TSimpleExpression;
    class TUnaryExpression;
    class TBinaryExpression;

    class IVisitor {
    public:
        virtual void visitSimpleExpression(const TSimpleExpression& expression) = 0;
        virtual void visitUnaryExpression(const TUnaryExpression& expression) = 0;
        virtual void visitBinaryExpression(const TBinaryExpression& expression) = 0;
    };

    class IExpression {
    public:
        virtual ~IExpression() = default;

        virtual void accept(IVisitor& visitor) const = 0;
    };

    using TExpressionPtr = TAtomicSharedPtr<IExpression>;

    class TSimpleExpression final : public IExpression {
    public:
        enum class EOp {
            Equal,
            NotEqual,
            Less,
            LessOrEqual,
            More,
            MoreOrEqual,
            Contains,
            StartsWith,
            EndsWith
        };

        using TFieldType = TString;
        using TValueType = NYT::TNode;

        TSimpleExpression(EOp, const TFieldType& field, const TValueType& value);

        ~TSimpleExpression() override final;

        void accept(IVisitor& visitor) const override final;

        EOp Op;
        TFieldType Field;
        TValueType Value;
    };

    class TUnaryExpression final : public IExpression {
    public:
        enum class EOp {
            Not
        };

        TUnaryExpression(EOp op, const TExpressionPtr& child);

        ~TUnaryExpression() override final;

        void accept(IVisitor& visitor) const override final;

        EOp Op;
        TExpressionPtr Child = nullptr;
    };

    class TBinaryExpression final : public IExpression {
    public:
        enum class EOp {
            And,
            Or
        };

        TBinaryExpression(EOp op, const TExpressionPtr& left, const TExpressionPtr& right);

        ~TBinaryExpression() override final;

        void accept(IVisitor& visitor) const override final;

        EOp Op;
        TExpressionPtr Left = nullptr;
        TExpressionPtr Right = nullptr;
    };

    TExpressionPtr MakeSimpleExpression(TSimpleExpression::EOp op, const TSimpleExpression::TFieldType& field, const TSimpleExpression::TValueType& value);
    TExpressionPtr MakeUnaryExpression(TUnaryExpression::EOp op, const TExpressionPtr& child);
    TExpressionPtr MakeBinaryExpression(TBinaryExpression::EOp op, const TExpressionPtr& left, const TExpressionPtr& right);
}
